// IRScopeDlg.cpp : implementation file
//

/*
    Copyright (C) 2007,2008  Kevin Timmerman

	irwidget [@t] compendiumarcana [d0t] com

	http://www.compendiumarcana.com/irwidget

	Versions 2.00 and 2.01 developed from Kevin Timmerman's version
	1.51 by	Graham Dixon.
	
	Additions in versions 2.00 Copyright (C) 2009 Graham Dixon.
	Additions in versions 2.01 Copyright (C) 2010 Graham Dixon.

	This program is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program.  If not, see <http://www.gnu.org/licenses/>.

	The modifications in version 2.00 include addition of the following 
	facilities:

	* Re-sizability of main window
	* Menu
	* Implementation of menu items not duplicated elsewhere
	* Settings saved in registry
	* Opening .ict files on command line
	* Association of .ict file extension with IRScope
	* Single instance opening
	* Autolocation of Tommy Tyler widgets
	* New columns Start and End in Decode window
	* Highlighting of waveform corresponding to selected decode
	* Version info in About box now taken from the Version resource

	These modifications make use of the following open source modules
	redistributed with this source code:

	* EasySize.h
	* GlobalFunctions.cpp, GlobalFunctions.h
	* singleinstance.cpp, singleinstance.h

	The modifications in version 2.01 include addition of the following 
	facilities:

	* Importing of signals in UEI Learned, Pronto or Timing List formats
	* Exporting of signals in UEI Learned, Pronto or Lintronic formats
	* Creation of a Pronto export based on the decode results with John
	  Fine's MakeHex program and its IRP files
	* A Summary display showing the decodes and timing data in both a raw
	  and analyzed form.  The analyzed data includes a representation of
	  the signal in IRP form that is independent of the DecodeIR result
	  and which works even when the protocol is not known.
	* Double-clicking a decode entry opens or re-opens a waveform window
	  and highlights the selected decode
	* Ability to annotate, delete and re-order entries in the decode list

	Import, export and analysis facilities make use of the following new DLLs
	distributed with this program:

	* ExchangeIR.dll by Graham Dixon
	* MakeHex.dll which is John Fine's MakeHex.exe converted to DLL format

*/

#include "stdafx.h"
#include "IRScope.h"
#include "IRScopeDlg.h"
#include "widget.h"
#include "audio.h"
#include "ftdi.h"
#include <math.h>
#include "IRNoteEdit.h"
#include "IRSummary.h"
#include "IRImport.h"
#include "GlobalFunctions.h"


#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif


CString GetExeByExtension(CString sExt)
{
	HKEY hkey;
	DWORD iSize;
	char sExtDesc[256];
	char sPath[1024];
	CString sCmd = _T("");
	if (RegOpenKeyEx(HKEY_CLASSES_ROOT, sExt,
			0, KEY_QUERY_VALUE, &hkey) == ERROR_SUCCESS)
	{
		iSize = sizeof(sExtDesc);
		if (RegQueryValueEx(hkey, _T(""), NULL, NULL,
					(LPBYTE)sExtDesc, &iSize) == ERROR_SUCCESS
				&&	(char*)sExtDesc[iSize-1] == 0 )
		{
			sCmd = _T(sExtDesc)+CString("\\Shell\\Open\\Command");
		}
		RegCloseKey(hkey);
		if (sCmd != _T("") && RegOpenKeyEx(HKEY_CLASSES_ROOT, sCmd,
			0, KEY_QUERY_VALUE, &hkey) == ERROR_SUCCESS)
		{
			iSize = sizeof(sPath);
			if (RegQueryValueEx(hkey, _T(""), NULL, NULL,
					(LPBYTE)sPath, &iSize) == ERROR_SUCCESS
				&&	(char*)sPath[iSize-1] == 0 )
			{
				sCmd = CString(sPath);
			}
			else
			{
				sCmd = _T("");
			}
			RegCloseKey(hkey);
		}
	}
	if (sCmd != _T(""))
	{
		if (sCmd.Find((TCHAR)('\"')) == 0)
		{
			// Extract quoted string
			sCmd = sCmd.Mid(1);
			sCmd = sCmd.Left(sCmd.Find((TCHAR)('\"')));
		}
		else if (sCmd.Find((TCHAR)('\"')) > 0)
		{
			// Extract string preceding first quote
			sCmd = sCmd.Left(sCmd.Find((TCHAR)('\"')));
		}

		if (sCmd.Find(CString("%1")) > 0)
		{
			// Extract string preceding %1 if present
			sCmd = sCmd.Left(sCmd.Find(CString("%1")));
		}

		iSize = ExpandEnvironmentStrings(sCmd, sPath, sizeof(sPath));
		if (iSize > 0)
		{
			sCmd = CString(sPath);
		}
	}
	return sCmd.Trim();			
}

/////////////////////////////////////////////////////////////////////////////
// CAboutDlg dialog used for App About

class CAboutDlg : public CDialog
{
public:
	CAboutDlg();

// Dialog Data
	//{{AFX_DATA(CAboutDlg)
	enum { IDD = IDD_ABOUTBOX };
	//}}AFX_DATA

	// ClassWizard generated virtual function overrides
	//{{AFX_VIRTUAL(CAboutDlg)
	protected:
	virtual void DoDataExchange(CDataExchange* pDX);    // DDX/DDV support
	//}}AFX_VIRTUAL

// Implementation
protected:
	//{{AFX_MSG(CAboutDlg)
	virtual BOOL OnInitDialog();
	//}}AFX_MSG
	DECLARE_MESSAGE_MAP()
};

CAboutDlg::CAboutDlg() : CDialog(CAboutDlg::IDD)
{
	//{{AFX_DATA_INIT(CAboutDlg)
	//}}AFX_DATA_INIT
}

void CAboutDlg::DoDataExchange(CDataExchange* pDX)
{
	CDialog::DoDataExchange(pDX);
	//{{AFX_DATA_MAP(CAboutDlg)
	//}}AFX_DATA_MAP
}

BOOL CAboutDlg::OnInitDialog() 
{
	CDialog::OnInitDialog();
	
	HANDLE h=FindResource(NULL,MAKEINTRESOURCE(IDR_LICENSE),"LICENSE");
	if(h) h=LoadResource(NULL,FindResource(NULL,MAKEINTRESOURCE(IDR_LICENSE),"LICENSE"));
	if(h) {
		LPCSTR s=static_cast<LPCSTR>(LockResource(h));
		if(s) GetDlgItem(IDC_LICENSE)->SetWindowText(s);
		FreeResource(h);
	}
	GetDlgItem(IDC_VERSION)->SetWindowText("IRScope version " +
		CGlobalFunctions::GetVersionInfo(NULL, "ProductVersion"));
	
	return TRUE;  // return TRUE unless you set the focus to a control
	              // EXCEPTION: OCX Property Pages should return FALSE
}

BEGIN_MESSAGE_MAP(CAboutDlg, CDialog)
	//{{AFX_MSG_MAP(CAboutDlg)
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
// CIRScopeDlg dialog

CIRScopeDlg::CIRScopeDlg(CWnd* pParent /*=NULL*/)
	: CDialog(CIRScopeDlg::IDD, pParent)
	, m_count(0)
	, m_times(NULL)
	, m_freq(0)
	, m_errlimit(0)
	, m_counts(NULL)
	, m_currentWnd(NULL)
	, m_bRestore(0)
	, m_AutoWidget(FALSE)
	, m_NoBeeps(FALSE)
	, m_HideSavepanel(FALSE)
	, m_bAutolocated(FALSE)
	, m_sAppFolder(_T(""))
	, m_iThumbScale(0)
	, m_iExportMode(0)
	, m_SelectOnCapture(FALSE)
	, m_iIRPformat(0)
{
	//{{AFX_DATA_INIT(CIRScopeDlg)
	m_Duration = 700;
	m_Port = _T("COM5");
	m_Hardware = 0;
	m_SaveICT =	FALSE;
	m_SaveTVBG = FALSE;
	m_SaveWav = FALSE;
	m_AskName = FALSE;
	m_ClearBeforeCapture = FALSE;
	m_ShowWaveform = TRUE;
	//}}AFX_DATA_INIT
	// Note that LoadIcon does not require a subsequent DestroyIcon in Win32
	m_hIcon = AfxGetApp()->LoadIcon(IDR_MAINFRAME);
	m_hAccel = ::LoadAccelerators(AfxGetInstanceHandle(),
		MAKEINTRESOURCE (IDR_IR_ACCEL));
}

void CIRScopeDlg::DoDataExchange(CDataExchange* pDX)
{
	CDialog::DoDataExchange(pDX);
	//{{AFX_DATA_MAP(CIRScopeDlg)
	DDX_Control(pDX, IDC_DECODE, m_Decode);
	DDX_Text(pDX, IDC_DURATION, m_Duration);
	DDV_MinMaxUInt(pDX, m_Duration, 500, 15000);
	DDX_CBString(pDX, IDC_PORT, m_Port);
	DDX_CBIndex(pDX, IDC_HARDWARE, m_Hardware);
	DDX_Check(pDX, IDC_SAVE_ICT, m_SaveICT);
	DDX_Check(pDX, IDC_SAVE_TVBG, m_SaveTVBG);
	DDX_Check(pDX, IDC_SAVE_WAV, m_SaveWav);
	DDX_Check(pDX, IDC_ASK_NAME, m_AskName);
	DDX_Check(pDX, IDC_CLEAR_BEFORE, m_ClearBeforeCapture);
	DDX_Check(pDX, IDC_SHOW_WAVE, m_ShowWaveform);
	//}}AFX_DATA_MAP
	DDX_Control(pDX, IDC_AUTOSAVE, m_AutoSaveNote);
}

BEGIN_MESSAGE_MAP(CIRScopeDlg, CDialog)
	//{{AFX_MSG_MAP(CIRScopeDlg)
	ON_WM_SYSCOMMAND()
	ON_WM_PAINT()
	ON_WM_QUERYDRAGICON()
	ON_BN_CLICKED(IDC_CAPTURE, OnCapture)
	ON_BN_CLICKED(IDC_VIEW, OnView)
	ON_BN_CLICKED(IDC_CLEAR_DECODE, OnClearDecode)
	ON_WM_DESTROY()
	//}}AFX_MSG_MAP
	// For EasySize - start
	ON_WM_SIZE()
	ON_WM_SIZING()
	// For EasySize - end
	ON_WM_CREATE()
	ON_COMMAND(ID_FILE_OPEN, &CIRScopeDlg::OnFileOpen)
	ON_COMMAND(ID_FILE_SAVE, &CIRScopeDlg::OnFileSave)

	ON_MESSAGE(WM_OPEN_DLG_FILE, OnOpenICTFile)
	ON_WM_INITMENUPOPUP()
	ON_COMMAND(ID_FILE_CLEAR, &CIRScopeDlg::OnFileClear)
	ON_COMMAND(ID_FILE_CAPTURE, &CIRScopeDlg::OnFileCapture)
	ON_COMMAND(ID_FILE_SETSAVEFOLDER, &CIRScopeDlg::OnFileSetsavefolder)
	ON_COMMAND(ID_APP_ABOUT, &CIRScopeDlg::OnAppAbout)
	ON_COMMAND(ID_FILE_SAVE_AS, &CIRScopeDlg::OnFileSaveAs)
	ON_NOTIFY(NM_CLICK, IDC_DECODE, &CIRScopeDlg::OnNMClickDecode)
	ON_CBN_SELCHANGE(IDC_HARDWARE, &CIRScopeDlg::OnCbnSelchangeHardware)
	ON_COMMAND(ID_AUTOTYPES_ICT, &CIRScopeDlg::OnAutotypesIct)
	ON_COMMAND(ID_AUTOTYPES_WAV, &CIRScopeDlg::OnAutotypesWav)
	ON_COMMAND(ID_AUTOTYPES_TVBG, &CIRScopeDlg::OnAutotypesTvbg)
	ON_COMMAND_RANGE(ID_AUTOMODES_AUTO, ID_AUTOMODES_PROMPT, &CIRScopeDlg::OnAutomodes)
	ON_BN_CLICKED(IDC_SAVE_ICT, &CIRScopeDlg::OnBnClickedSaveIct)
	ON_BN_CLICKED(IDC_SAVE_WAV, &CIRScopeDlg::OnBnClickedSaveWav)
	ON_BN_CLICKED(IDC_SAVE_TVBG, &CIRScopeDlg::OnBnClickedSaveTvbg)
	ON_BN_CLICKED(IDC_ASK_NAME, &CIRScopeDlg::OnBnClickedAskName)
	ON_COMMAND(ID_AUTOWIDGET, &CIRScopeDlg::OnAutowidget)
	ON_COMMAND(ID_OPTIONS_NOBEEPS, &CIRScopeDlg::OnOptionsNobeeps)
	ON_COMMAND(ID_OPTIONS_HIDESAVEPANEL, &CIRScopeDlg::OnOptionsHidesavepanel)
	ON_NOTIFY(NM_RCLICK, IDC_DECODE, &CIRScopeDlg::OnNMRClickDecode)
//	ON_COMMAND(ID_FILE_SAVESELECTED, &CIRScopeDlg::OnFileSaveSelected)
	ON_COMMAND(ID_FILE_SAVESELECTEDAS, &CIRScopeDlg::OnFileSaveSelectedAs)
	ON_COMMAND(ID_HELP_DECODEIR_HTML, &CIRScopeDlg::OnHelpDecodeirHtml)
	ON_COMMAND_RANGE(ID_THUMB_FIXEDRES, ID_THUMB_ENTIREWAVE, &CIRScopeDlg::OnThumbScale)
	ON_COMMAND(ID_FILE_SETEXPORTFOLDER, &CIRScopeDlg::OnFileSetexportfolder)
	ON_BN_CLICKED(IDC_BN_EXPORT, &CIRScopeDlg::OnBnClickedBnExport)
	ON_COMMAND(ID_EXPORT_EXPORT, &CIRScopeDlg::OnExportExport)
	ON_COMMAND(ID_EXPORT_EXPORTSELECTED, &CIRScopeDlg::OnExportExportselected)
	ON_COMMAND_RANGE(ID_EXPORTMODE_UEILEARNED, ID_EXPORTMODE_LINTRONIC, &CIRScopeDlg::OnExportMode)
	ON_NOTIFY(NM_DBLCLK, IDC_DECODE, &CIRScopeDlg::OnNMDblclkDecode)
	ON_BN_CLICKED(IDC_BN_DELETE, &CIRScopeDlg::OnBnClickedBnDelete)
	ON_COMMAND(ID_FILE_DELETESELECTED, &CIRScopeDlg::OnBnClickedBnDelete) 
	ON_BN_CLICKED(IDC_BN_NOTE, &CIRScopeDlg::OnBnClickedBnNote)
	ON_COMMAND(ID_FILE_ADDNOTETOSELECTED, &CIRScopeDlg::OnBnClickedBnNote)
	ON_BN_CLICKED(IDC_BN_UP, &CIRScopeDlg::OnBnClickedBnUp)
	ON_COMMAND(ID_EDIT_MOVESELECTEDUP, &CIRScopeDlg::OnBnClickedBnUp)
	ON_BN_CLICKED(IDC_BN_DOWN, &CIRScopeDlg::OnBnClickedBnDown)
	ON_COMMAND(ID_EDIT_MOVESELECTEDDOWN, &CIRScopeDlg::OnBnClickedBnDown)
	ON_BN_CLICKED(IDC_BN_SAVE, &CIRScopeDlg::OnBnClickedBnSave)
	ON_BN_CLICKED(IDC_BN_SAVEAS, &CIRScopeDlg::OnBnClickedBnSaveas)
	ON_COMMAND(ID_SELECT_SELECTALL, &CIRScopeDlg::SelectAll)
	ON_COMMAND(ID_SELECT_DESELECTALL, &CIRScopeDlg::DeselectAll)
	ON_COMMAND(ID_FILE_SETIRPFOLDER, &CIRScopeDlg::OnFileSetirpfolder)
	ON_COMMAND(ID_OPTIONS_SELECTONCAPTURE, &CIRScopeDlg::OnOptionsSelectOnCapture)
	ON_COMMAND(ID_FILE_SUMMARY, &CIRScopeDlg::OnFileSummary)
	ON_BN_CLICKED(IDC_BN_SUMMARY, &CIRScopeDlg::OnFileSummary)
	ON_BN_CLICKED(IDC_BN_IMPORT, &CIRScopeDlg::OnBnClickedBnImport)
	ON_COMMAND(ID_FILE_IMPORT, &CIRScopeDlg::OnBnClickedBnImport)
	ON_COMMAND(ID_OPTIONS_COMMONFOLDER, &CIRScopeDlg::OnOptionsCommonfolder)
	ON_COMMAND(ID_EDIT_SELECTALL, &CIRScopeDlg::SelectAll)
	ON_COMMAND(ID_EDIT_DESELECTALL, &CIRScopeDlg::DeselectAll)
	ON_COMMAND(ID_EDIT_CLEARALL, &CIRScopeDlg::OnClearDecode)
	ON_COMMAND(ID_OPTIONS_DISABLEDECODER, &CIRScopeDlg::OnOptionsDisabledecoder)
	ON_COMMAND_RANGE(ID_IRPDATAFORMAT_BINARY, ID_IRPDATAFORMAT_HEXS, &CIRScopeDlg::OnIrpdataformat)
	ON_BN_CLICKED(IDC_SAVEALLAS, &CIRScopeDlg::OnBnClickedSaveallas)
	ON_COMMAND(ID_FILE_SAVEALLAS, &CIRScopeDlg::OnFileSaveallas)
END_MESSAGE_MAP()

BEGIN_EASYSIZE_MAP(CIRScopeDlg)
	EASYSIZE(IDC_DECODE, ES_BORDER, ES_BORDER, ES_BORDER, ES_BORDER, 0)
	EASYSIZE(IDC_DECODE_STRUCT, ES_BORDER, ES_KEEPSIZE, ES_BORDER, ES_BORDER, 0) 
	EASYSIZE(IDC_CLEAR_DECODE, ES_BORDER, ES_KEEPSIZE, ES_KEEPSIZE, ES_BORDER, 0)
	EASYSIZE(IDC_CLEAR_BEFORE, ES_BORDER, ES_KEEPSIZE, ES_KEEPSIZE, ES_BORDER, 0)
	EASYSIZE(IDC_SHOW_WAVE, ES_BORDER, ES_KEEPSIZE, ES_KEEPSIZE, ES_BORDER, 0)
	EASYSIZE(IDC_AUTOSAVE, ES_KEEPSIZE, ES_KEEPSIZE, ES_BORDER, ES_BORDER, 0)
	EASYSIZE(IDC_VIEW, ES_KEEPSIZE, ES_BORDER, ES_BORDER, ES_KEEPSIZE, 0)
	EASYSIZE(IDC_CAPTURE, ES_KEEPSIZE, ES_BORDER, ES_BORDER, ES_KEEPSIZE, 0)
	EASYSIZE(IDC_STATIC_SIG, ES_BORDER, ES_KEEPSIZE, ES_KEEPSIZE, ES_BORDER, 0)
	EASYSIZE(IDC_BN_NOTE, ES_KEEPSIZE, ES_BORDER, ES_BORDER, ES_KEEPSIZE, 0)
//	EASYSIZE(IDC_BN_SAVE, ES_KEEPSIZE, ES_BORDER, ES_BORDER, ES_KEEPSIZE, 0)
	EASYSIZE(IDC_BN_SAVEAS, ES_KEEPSIZE, ES_BORDER, ES_BORDER, ES_KEEPSIZE, 0)
	EASYSIZE(IDC_BN_EXPORT, ES_KEEPSIZE, ES_BORDER, ES_BORDER, ES_KEEPSIZE, 0)
	EASYSIZE(IDC_BN_DELETE, ES_KEEPSIZE, ES_BORDER, ES_BORDER, ES_KEEPSIZE, 0)
	EASYSIZE(IDC_BN_UP, ES_KEEPSIZE, ES_BORDER, ES_BORDER, ES_KEEPSIZE, 0)
	EASYSIZE(IDC_BN_DOWN, ES_KEEPSIZE, ES_BORDER, ES_BORDER, ES_KEEPSIZE, 0)
	EASYSIZE(IDC_STATIC_ACT, ES_KEEPSIZE, ES_BORDER, ES_BORDER, ES_KEEPSIZE, 0)
	EASYSIZE(IDC_BN_IMPORT, ES_KEEPSIZE, ES_BORDER, ES_BORDER, ES_KEEPSIZE, 0)
	EASYSIZE(IDC_BN_SUMMARY, ES_BORDER, ES_KEEPSIZE, ES_KEEPSIZE, ES_BORDER, 0)
	EASYSIZE(IDC_STATIC_USERNOTE, ES_BORDER, ES_KEEPSIZE, ES_KEEPSIZE, ES_BORDER, 0)
	EASYSIZE(IDC_DECODE_IRP, ES_BORDER, ES_KEEPSIZE, ES_BORDER, ES_BORDER, 0)
	EASYSIZE(IDC_STATIC_IRP, ES_BORDER, ES_KEEPSIZE, ES_KEEPSIZE, ES_BORDER, 0)
	EASYSIZE(IDC_SAVEALLAS, ES_KEEPSIZE, ES_BORDER, ES_BORDER, ES_KEEPSIZE, 0)
END_EASYSIZE_MAP

/////////////////////////////////////////////////////////////////////////////
// CIRScopeDlg message handlers

BOOL CIRScopeDlg::OnInitDialog()
{
	CDialog::OnInitDialog();

	// Add "About..." menu item to system menu.

	// IDM_ABOUTBOX must be in the system command range.
	ASSERT((IDM_ABOUTBOX & 0xFFF0) == IDM_ABOUTBOX);
	ASSERT(IDM_ABOUTBOX < 0xF000);

	CMenu* pSysMenu = GetSystemMenu(FALSE);
	CString strAboutMenu;
	strAboutMenu.LoadString(IDS_ABOUTBOX);
	if (!strAboutMenu.IsEmpty())
	{
		pSysMenu->AppendMenu(MF_SEPARATOR);
		pSysMenu->AppendMenu(MF_STRING, IDM_ABOUTBOX, strAboutMenu);
	}

	// Set the icon for this dialog.  The framework does this automatically
	//  when the application's main window is not a dialog
	SetIcon(m_hIcon, TRUE);			// Set big icon
	SetIcon(m_hIcon, FALSE);		// Set small icon
	
	// Add extra initialization here
	HKEY hkey;
	DWORD iSize;
	int aiDefColWidths[] = {20, 64, 64, 64, 48, 64, 96, 96, 64, 64, 64};
	int iColCount = sizeof(aiDefColWidths)/sizeof(int);
	int aiColWidths[16];	// assumes number of cols will never exceed 16
	int aCtrlID[] = CTRL_LIST;
	int aiCtrlPos[50];
	RECT rect;

	UINT aOptList[] = OPT_LIST;
	int aOptSub[] = OPT_SUB;
	BOOL* aOptVar[] = OPT_VAR;
	int iOptCount = sizeof(aOptList)/sizeof(UINT);
	DWORD dwState;

	// Control positions only taken from registry if DecodeIR found
	if (RegOpenKeyEx(HKEY_CURRENT_USER, REG_KEY_SET,
			0, KEY_QUERY_VALUE, &hkey) == ERROR_SUCCESS)
	{
		iSize = sizeof(aiColWidths);
		if ( RegQueryValueEx(hkey, TEXT("DecodeCols"), NULL, NULL,
					(LPBYTE)aiColWidths, &iSize ) == ERROR_SUCCESS
				// check stored col count = actual col count
				&&	iSize == iColCount * sizeof(int) )
		{
			for (int i = 0; i < iColCount; i++)
			{
				if (aiColWidths[i] >= 0)
					aiDefColWidths[i] = aiColWidths[i];
			} 
		}
		iSize = sizeof(aiCtrlPos);
		if ( (m_hDecodeLib) && (m_bRestore) && 
			  RegQueryValueEx(hkey, TEXT("ControlPos"), NULL, NULL,
					(LPBYTE)aiCtrlPos, &iSize ) == ERROR_SUCCESS ) {
			for (int i=0; i<(int)(iSize/(2*sizeof(int))); i++)
			{
				GetDlgItem(aCtrlID[i])->GetWindowRect(&rect);
				ScreenToClient(&rect);
				if (i>2) rect.left += aiCtrlPos[2*i] - rect.right;
				rect.right = aiCtrlPos[2*i];
				if (i>0) rect.top += aiCtrlPos[2*i+1] - rect.bottom;
				rect.bottom = aiCtrlPos[2*i+1];
				GetDlgItem(aCtrlID[i])->MoveWindow(&rect);
			}
		}
		iSize = sizeof(DWORD);
		if ( RegQueryValueEx(hkey, TEXT("ExportMode"), NULL, NULL,
					(LPBYTE)&dwState, &iSize ) != ERROR_SUCCESS ) {
			dwState = EXPORT_DEFAULT;
		}
		if ( ! m_hExchangeLib && m_hMakeHexLib && 
				dwState != ID_EXPORTMODE_PRONTODEC - ID_EXPORTMODE_UEILEARNED ) {
			OnExportMode(ID_EXPORTMODE_PRONTODEC);
		} else if ( ! m_hMakeHexLib && m_hExchangeLib && 
				dwState == ID_EXPORTMODE_PRONTODEC - ID_EXPORTMODE_UEILEARNED) {
			OnExportMode(ID_EXPORTMODE_UEILEARNED + EXPORT_DEFAULT);
		} else OnExportMode(ID_EXPORTMODE_UEILEARNED + dwState);

		iSize = sizeof(DWORD);
		if ( RegQueryValueEx(hkey, TEXT("IRPFormat"), NULL, NULL,
					(LPBYTE)&dwState, &iSize ) != ERROR_SUCCESS ) {
			dwState = IRPDATAFORMAT_DEFAULT;
		}
		OnIrpdataformat(ID_IRPDATAFORMAT_BINARY + dwState);

		iSize = sizeof(DWORD);
		if ( RegQueryValueEx(hkey, TEXT("Options"), NULL, NULL,
					(LPBYTE)&dwState, &iSize ) != ERROR_SUCCESS ) {
			dwState = OPT_DEFAULT;
		}
		RegCloseKey(hkey);

		for (int i=0; i<iOptCount; i++) {
			switch (aOptSub[i])
			{
				case -2:
					if (aOptVar[i] != NULL) *aOptVar[i] = dwState>>i & 1;
					break;
				case -1:
					// i = 8 is Hide Savepanel, which is toggled again below
					// i = 12 is Common Output Folder, which is toggled again below
					GetMenu()->GetSubMenu(OPT_MENU)->CheckMenuItem(aOptList[i], ((dwState>>i&1)^(i==8 || i==12) ? 
							MF_CHECKED : MF_UNCHECKED)| MF_BYCOMMAND);
					if (aOptVar[i] != NULL) *aOptVar[i] = dwState>>i & 1;
					break;
				case 0:
					ToggleMenuOption(aOptList[i], aOptVar[i], 0, dwState>>i & 1);
					break;
				case 1:
					OnAutomodes((dwState>>i & 1) ? ID_AUTOMODES_PROMPT : ID_AUTOMODES_AUTO);
					break;
				case 7:
					OnThumbScale((dwState>>i++ & 1) ? ID_THUMB_FIXEDRES :
						(dwState>>i & 1) ? ID_THUMB_LIMITEDDUR : ID_THUMB_ENTIREWAVE );
					break;
			}
		}
		UpdateData(false);
		OnOptionsHidesavepanel(); // Toggle them back and act on result
		OnOptionsCommonfolder();
	}

	ASSERT (iColCount >= 9);

	m_Decode.InsertColumn(0,"#",LVCFMT_LEFT,aiDefColWidths[0]);
	m_Decode.InsertColumn(1,"Frequency",LVCFMT_LEFT,aiDefColWidths[1]);
	m_Decode.InsertColumn(2,"Protocol",LVCFMT_LEFT,aiDefColWidths[2]);
	m_Decode.InsertColumn(3,"Device",LVCFMT_LEFT,aiDefColWidths[3]);
	m_Decode.InsertColumn(4,"OBC",LVCFMT_LEFT,aiDefColWidths[4]);
	m_Decode.InsertColumn(5,"Hex",LVCFMT_LEFT,aiDefColWidths[5]);
	m_Decode.InsertColumn(6,"Note",LVCFMT_LEFT,aiDefColWidths[6]);
	m_Decode.InsertColumn(7,"Misc",LVCFMT_LEFT,aiDefColWidths[7]);
	m_Decode.InsertColumn(8,"Start",LVCFMT_LEFT,aiDefColWidths[8]);
	m_Decode.InsertColumn(9,"End",LVCFMT_LEFT,aiDefColWidths[9]);
	m_Decode.InsertColumn(10,"Error",LVCFMT_LEFT,aiDefColWidths[10]);

	m_Decode.SetExtendedStyle(LVS_EX_FULLROWSELECT);
	m_Decode.ModifyStyle(LVS_SINGLESEL, 0, 0);

	CString verStr = "";
	if(m_hDecodeLib) {
		m_GetDecodeVersion=reinterpret_cast<typeGetVersion>(GetProcAddress(m_hDecodeLib,"Version"));
		m_DecodeIR=reinterpret_cast<typeDecodeIR>(GetProcAddress(m_hDecodeLib,"DecodeIR"));
		if(m_DecodeIR) {
			char version[64];
			m_GetDecodeVersion(version);
			verStr = CString("IR Scope v" +
				CGlobalFunctions::GetVersionInfo(NULL, "ProductVersion") +
				" (DecodeIR v")+version;
			if (!m_hExchangeLib && !m_hMakeHexLib) verStr += ")";
		}
	} else {
		m_GetDecodeVersion=NULL;
		m_DecodeIR=NULL;
		RECT rectDecode, rectCapture, rectView, rectSaveAllAs;
		GetWindowRect(&rect);
		GetDlgItem(IDC_DECODE)->GetWindowRect(&rectDecode);
		GetDlgItem(IDC_CAPTURE)->GetWindowRect(&rectCapture);
		GetDlgItem(IDC_VIEW)->GetWindowRect(&rectView);
		GetDlgItem(IDC_SAVEALLAS)->GetWindowRect(&rectSaveAllAs);
		int iViewHeight = rectView.bottom - rectView.top;
		rectView.right = rectCapture.right;
		rectView.bottom += iViewHeight/2;
		rectCapture.top += iViewHeight/2;
		rectCapture.bottom = rectSaveAllAs.bottom;
		ScreenToClient(&rectView);
		ScreenToClient(&rectCapture);
		GetDlgItem(IDC_VIEW)->MoveWindow(&rectView);
		GetDlgItem(IDC_CAPTURE)->MoveWindow(&rectCapture);
		GetDlgItem(IDC_DECODE)->ShowWindow(SW_HIDE);
		GetDlgItem(IDC_DECODE_STRUCT)->ShowWindow(SW_HIDE);
		GetDlgItem(IDC_DECODE_IRP)->ShowWindow(SW_HIDE);
		for (int i = 5; i < sizeof(aCtrlID)/sizeof(int); i++)
			GetDlgItem(aCtrlID[i])-> ShowWindow(SW_HIDE);
		rect.bottom=rectDecode.top;
		rect.right=rect.left+450;
		// Set Savepanel visible and gray the Hide Savepanel option
		if (m_HideSavepanel) OnOptionsHidesavepanel();
		for (int i=0; i<HELP_MENU; i++) GetMenu()->EnableMenuItem(i, MF_GRAYED|MF_BYPOSITION);
		GetMenu()->GetSubMenu(OPT_MENU)->EnableMenuItem(ID_OPTIONS_HIDESAVEPANEL, MF_GRAYED|MF_BYCOMMAND);
		// Set Show Waveform
		m_ShowWaveform = true;
		UpdateData(false);
	}

	if ( !(m_hExchangeLib || m_hMakeHexLib) )
	{
		CMenu* menu = GetMenu();
		GetDlgItem(IDC_BN_EXPORT)->EnableWindow(false);
		menu->EnableMenuItem(EXPORT_MENU, MF_GRAYED|MF_BYPOSITION);
		DrawMenuBar();
	}

	if(m_hDecodeLib && m_hExchangeLib) {
		char version[64];
		version[0] = 0;
		m_GetExchangeVersion=reinterpret_cast<typeGetVersion>(GetProcAddress(m_hExchangeLib,"Version"));
		m_MakePronto=reinterpret_cast<typeMakePronto>(GetProcAddress(m_hExchangeLib,"MakePronto"));
		m_ReadPronto=reinterpret_cast<typeReadPronto>(GetProcAddress(m_hExchangeLib,"ReadPronto"));
		m_FindRepeat=reinterpret_cast<typeFindRepeat>(GetProcAddress(m_hExchangeLib,"FindRepeat"));
		m_Analyze=reinterpret_cast<typeAnalyze>(GetProcAddress(m_hExchangeLib,"Analyze"));
		m_MakeUEILearned=reinterpret_cast<typeMakeUEILearned>(GetProcAddress(m_hExchangeLib,"MakeUEILearned"));
		m_ReadUEILearned=reinterpret_cast<typeReadUEILearned>(GetProcAddress(m_hExchangeLib,"ReadUEILearned"));
		m_MakeLintronic=reinterpret_cast<typeMakeLintronic>(GetProcAddress(m_hExchangeLib,"MakeLintronic"));
		m_MakeHex=reinterpret_cast<typeMakeHex>(GetProcAddress(m_hMakeHexLib,"MakeHex"));

		if(m_GetExchangeVersion) {
			m_GetExchangeVersion(version);
			verStr += CString(", ExchangeIR v")+version;
			if (!m_hMakeHexLib) verStr += ")";
		}
	}
	else {
		CMenu* menu = GetMenu()->GetSubMenu(EXPORT_MENU);
		menu->EnableMenuItem(ID_EXPORTMODE_UEILEARNED, MF_BYCOMMAND | MF_GRAYED);
		menu->EnableMenuItem(ID_EXPORTMODE_PRONTOSIG, MF_BYCOMMAND | MF_GRAYED);
		menu->EnableMenuItem(ID_EXPORTMODE_LINTRONIC, MF_BYCOMMAND | MF_GRAYED);
		GetDlgItem(IDC_DECODE_IRP)->EnableWindow(false);
		GetDlgItem(IDC_STATIC_IRP)->EnableWindow(false);
		GetDlgItem(IDC_DECODE_IRP)->SetWindowText("N/A");
	}

	if(m_hDecodeLib && m_hMakeHexLib) {
		char version[64];
		version[0] = 0;
		m_GetMakeHexVersion=reinterpret_cast<typeGetVersion>(GetProcAddress(m_hMakeHexLib,"Version"));

		if(m_GetMakeHexVersion) {
			m_GetMakeHexVersion(version);
			verStr += CString(", MakeHex v")+version+")";
		}
	}
	else {
		CMenu* menu = GetMenu()->GetSubMenu(EXPORT_MENU);
		menu->EnableMenuItem(ID_EXPORTMODE_PRONTODEC, MF_BYCOMMAND | MF_GRAYED);
	}

	if (verStr != "") SetWindowText(verStr);

	CFont font;
	LOGFONT lf;
	m_AutoSaveNote.GetFont()->GetLogFont(&lf);
	lf.lfWeight = FW_BOLD;
	if ( font.CreateFontIndirect(&lf) )
	{
		m_AutoSaveNote.SetFont(&font);
		font.DeleteObject();
	}

	INIT_EASYSIZE;	
	if(! m_hDecodeLib)
	{
		// These need to be after INIT_EASYSIZE
		MoveWindow(&rect,FALSE);
		ModifyStyle(WS_THICKFRAME, WS_DLGFRAME, SWP_FRAMECHANGED);
	}

	SetAvailablePorts(m_Hardware);

	if (command_line_filename != "")
		OpenICTFile(command_line_filename);

	return TRUE;  // return TRUE  unless you set the focus to a control
}


void CIRScopeDlg::OnDestroy() 
{
	CDialog::OnDestroy();
	
	if(m_hDecodeLib) {
		FreeLibrary(m_hDecodeLib);
	}	

	if(m_hFTDILib) {
		FreeLibrary(m_hFTDILib);
	}
			
	HKEY hkey;
	DWORD dwDisp;
	DWORD iSize;
	int aiColWidths[16];	// assumes number of cols will never exceed 16
	HDITEM hdiDecode;
	hdiDecode.mask = HDI_WIDTH;
	int iColCount;
	WINDOWPLACEMENT wndpl;
	wndpl.length = sizeof(WINDOWPLACEMENT);
	RECT rect;
	int aCtrlID[] = CTRL_LIST;
	int iCtrlCount = sizeof(aCtrlID)/sizeof(int);
	int aiCtrlPos[50];  // assumes not more than 25 controls to position
	int normal = true;
	int iRevision = REVISION;
	DWORD dwOpts = 0;
	UINT aOptList[] = OPT_LIST;
	BOOL* aOptVar[] = OPT_VAR;
	int aOptSub[] = OPT_SUB;
	int iOptCount = sizeof(aOptList)/sizeof(UINT);
	CMenu* menu;
	UINT state;
	DWORD dwState;

	UpdateData();	// Update DDX values

	for (int i = 0; i < iOptCount; i++)
	{
		if (aOptSub[i] >= -1)
		{
			// Get Advanced menu
			menu = GetMenu()->GetSubMenu(OPT_MENU);
			if (aOptSub[i] >= 0) menu = menu->GetSubMenu(aOptSub[i]);
			state = menu->GetMenuState(aOptList[i], MF_BYCOMMAND);
			ASSERT(state != 0xFFFFFFFF);
			dwState = (state & MF_CHECKED) == MF_CHECKED;
		}
		else
		{
			// Checkbox option
			dwState = *aOptVar[i];
		}
		dwOpts |= dwState << i;
	}


	// Window and control positions and GUI revision number 
	// only saved if DecodeIR found and IRScope not minimized
	if (RegCreateKeyEx(HKEY_CURRENT_USER, REG_KEY_SET, 0, NULL,
			REG_OPTION_NON_VOLATILE, KEY_READ|KEY_WRITE, NULL, &hkey, 
			&dwDisp) == ERROR_SUCCESS) 
	{
		if ((m_hDecodeLib) && GetWindowPlacement(&wndpl))
		{
			normal = (wndpl.showCmd == SW_SHOWNORMAL);
			if (normal) {
				RegSetValueEx(hkey, TEXT("WindowPlacement"), 0, REG_BINARY,
					(PBYTE)&wndpl, wndpl.length);
			}
		}
		RegSetValueEx(hkey, TEXT("Duration"), 0, REG_DWORD, 
			(PBYTE)&m_Duration, sizeof(DWORD));
		RegSetValueEx(hkey, TEXT("Port"), 0, REG_SZ, 
			(PBYTE)m_Port.GetBuffer(0), m_Port.GetLength() + 1);
		RegSetValueEx(hkey, TEXT("Hardware"), 0, REG_DWORD, 
			(PBYTE)&m_Hardware, sizeof(DWORD));
		iSize = sizeof(DWORD);
		if ( !m_hDecodeLib && RegQueryValueEx(hkey, TEXT("Options"), NULL, NULL,
					(LPBYTE)&dwState, &iSize ) == ERROR_SUCCESS )
		{
			// Do not save options Show Waveform (6) and Hide Savepanel (8)
			// if DecodeIR not present and there is a saved value, as they are
			// set TRUE and FALSE respectively on initialization in this case
			dwOpts &= ~0x140;			// mask out current value
			dwOpts |= dwState & 0x140;	// put in existing saved value
		}
		RegSetValueEx(hkey, TEXT("Options"), 0, REG_DWORD, 
			(PBYTE)&dwOpts, sizeof(DWORD));
		RegSetValueEx(hkey, TEXT("ExportMode"), 0, REG_DWORD, 
			(PBYTE)&m_iExportMode, sizeof(DWORD));
		RegSetValueEx(hkey, TEXT("IRPFormat"), 0, REG_DWORD, 
			(PBYTE)&m_iIRPformat, sizeof(DWORD));

		if (strSavePath == _T(""))
		{
			RegDeleteKey(hkey, "SavePath");
		} 
		else
		{
			RegSetValueEx(hkey, TEXT("SavePath"), 0, REG_SZ, 
			(PBYTE)strSavePath.GetBuffer(0), strSavePath.GetLength()+1);
		}

		if (strExportPath == _T(""))
		{
			RegDeleteKey(hkey, "ExportPath");
		} 
		else
		{
			RegSetValueEx(hkey, TEXT("ExportPath"), 0, REG_SZ, 
			(PBYTE)strExportPath.GetBuffer(0), strExportPath.GetLength()+1);
		}

		if (strIRPPath == _T(""))
		{
			RegDeleteKey(hkey, "IRPPath");
		} 
		else
		{
			RegSetValueEx(hkey, TEXT("IRPPath"), 0, REG_SZ, 
			(PBYTE)strIRPPath.GetBuffer(0), strIRPPath.GetLength()+1);
		}

		iColCount = m_Decode.GetHeaderCtrl()->GetItemCount();
		for (int i=0; i<iColCount; i++)
		{
			if (m_Decode.GetHeaderCtrl()->GetItem(i, &hdiDecode))
				aiColWidths[i] = hdiDecode.cxy;
			else
				aiColWidths[i] = -1;
		}
		RegSetValueEx(hkey, TEXT("DecodeCols"), 0, REG_BINARY,
			(PBYTE)aiColWidths, iColCount * sizeof(int));

		if (m_hDecodeLib && normal) 
		{
			for (int i=0; i<2*iCtrlCount; i++)
			{
				GetDlgItem(aCtrlID[i/2])->GetWindowRect(&rect);
				ScreenToClient(&rect);
				aiCtrlPos[i] = rect.right;
				aiCtrlPos[++i] = rect.bottom;
			}
			RegSetValueEx(hkey, TEXT("ControlPos"), 0, REG_BINARY,
				(PBYTE)aiCtrlPos, 2 * iCtrlCount * sizeof(int));
			RegSetValueEx(hkey, TEXT("Revision"), 0, REG_DWORD, 
				(PBYTE)&iRevision, sizeof(DWORD));
		}
		// We have finished with the registry,
		// so liberate the resources we were using
		RegCloseKey(hkey);
	};

	// Closure of waveform windows moved here from IRScope.cpp as it is
	// needed before app saves the recent file list.
	CIrScopeWnd::CloseAll();
	// Free signal memory
	for (int i=0; i<m_Signals.size(); i++) m_Signals[i].done();
}

void CIRScopeDlg::OnSysCommand(UINT nID, LPARAM lParam)
{
	if ((nID & 0xFFF0) == IDM_ABOUTBOX)
	{
		CAboutDlg dlgAbout;
		dlgAbout.DoModal();
	}
	else
	{
		CDialog::OnSysCommand(nID, lParam);
	}
}

// If you add a minimize button to your dialog, you will need the code below
//  to draw the icon.  For MFC applications using the document/view model,
//  this is automatically done for you by the framework.

void CIRScopeDlg::OnPaint() 
{
	if (IsIconic())
	{
		CPaintDC dc(this); // device context for painting

		SendMessage(WM_ICONERASEBKGND, (WPARAM) dc.GetSafeHdc(), 0);

		// Center icon in client rectangle
		int cxIcon = GetSystemMetrics(SM_CXICON);
		int cyIcon = GetSystemMetrics(SM_CYICON);
		CRect rect;
		GetClientRect(&rect);
		int x = (rect.Width() - cxIcon + 1) / 2;
		int y = (rect.Height() - cyIcon + 1) / 2;

		// Draw the icon
		dc.DrawIcon(x, y, m_hIcon);
	}
	else
	{
		CDialog::OnPaint();
	}
}

// The system calls this to obtain the cursor to display while the user drags
//  the minimized window.
HCURSOR CIRScopeDlg::OnQueryDragIcon()
{
	return (HCURSOR) m_hIcon;
}

void CIRScopeDlg::OnCapture() 
{
	if(!UpdateData())
		return;

	CWaitCursor wait;

	if(m_ClearBeforeCapture) {
		OnClearDecode();
	}
	
	int err=OpenPort(m_Port,m_Hardware);

	CString s;
	if(err) {
		s.Format("Unable to open %s.\n\nMake sure the correct COM port is selected.",m_Port);
		AfxMessageBox(s);
		return;
	}

	const UINT sample_byte_count = m_Duration * 1000 / 100;

	BYTE* data=new BYTE[sample_byte_count];
	if(!data) return;

	if (! m_NoBeeps) MessageBeep(MB_OK);

	const UINT read=ReadPort(sample_byte_count,data,(m_Hardware<2) ? m_Duration+400 : m_Duration);
	TRACE("%i Bytes Read\n",read);
	ClosePort();

	if((m_Hardware<2)?read!=sample_byte_count:read==0) {
		if (read) {
			s.Format("Expected %i bytes, only got %i",sample_byte_count,read);
		}
		else {
			// Conditionals distinguish between an Autolocated widget, which must have
			// correct COM port selected and will be a Tommy Tyler widget with a white
			// LED, and a non-Autolocated widget, which can be one of Tommy's but could
			// also be home-made.
			s = "No data received from IR Widget.\n\n";
			if (!m_bAutolocated) {
				s += "Make sure the correct COM port is selected.\n\n";
			}
			s += "Press remote button within 5 seconds\nof pressing Capture! button";
			if (m_bAutolocated) {	
				s += " and make\nsure signal causes white LED to flash";
			}
			s += ".";
		}
		MessageBox(s);
		if(read<100) {
			delete[] data;
			return;
		}
	}

	if (m_counts != NULL) delete[] m_counts;	
	if (m_times != NULL) delete[] m_times;		
	m_counts = NULL;	
	m_times = NULL;	
	m_freq = 0;		

	err = (m_Hardware<3) ?
		ProcessPulseData(read,data,m_count,m_times,m_counts,(m_Hardware==2)?139:100,m_freq)
		: ProcessTimeData(read,data,m_count,m_times);

	if(err==0) {
		CString file_name, path_name;
		file_name=CTime::GetCurrentTime().Format("IR%Y%m%d%H%M%S");
		if (strSavePath == _T("")) path_name = file_name;
		else path_name = strSavePath+"\\"+file_name;

		BOOL save=TRUE;
		if(m_AskName && (m_SaveICT || m_SaveTVBG || m_SaveWav)) {
			CFileDialog dlg(FALSE,NULL,file_name,OFN_HIDEREADONLY|OFN_PATHMUSTEXIST);
			if(dlg.DoModal()==IDOK) {
				path_name=dlg.GetPathName();
				file_name = path_name.Right(path_name.GetLength() - path_name.ReverseFind((TCHAR)('\\')) - 1);
			} else {
				save=FALSE;
			}
		}
		if(save && (m_SaveICT || m_SaveTVBG || m_SaveWav)) {
			if(m_SaveICT)
				WriteFile(path_name+".ict",m_count,m_times,m_counts,m_freq);
			if(m_SaveTVBG)
				WriteTVBGFile(path_name+"tvbg.c",m_count,m_times,m_freq);
			if(m_SaveWav)
				WriteWavFile(path_name+".wav",m_count,m_times,m_freq);
		}

		int item = m_Signals.size();
		m_Signals.resize(item+1);
		m_Signals[item] = Signal(NULL, m_times, m_counts, m_count, m_freq);
		m_UserNotes.clear();

		if(m_ShowWaveform) {
			if (save && m_SaveICT) file_name += ".ict";
			else file_name = "IR Scope";
			Decode(m_count,m_times,m_counts,m_freq,item,true);
			m_currentWnd = CreateWaveform(file_name, item);
		}
		else {
			m_currentWnd =  NULL;
			Decode(m_count,m_times,m_counts,m_freq,item,true);
		}
	} else if(err==1) {
		MessageBox("No signal found");
		m_counts = NULL;
	} else m_counts = NULL;
	 
	delete[] data;
}



UINT CIRScopeDlg::WriteTVBGFile(LPCSTR filename, int count, int *times, UINT freq)
{
	if(!count) return 0;

	CFile file;
	if(file.Open(filename,CFile::modeWrite | CFile::modeCreate)) {
		CString s("const struct powercode xCode PROGMEM = {\r\n");
		file.Write(s,s.GetLength());
		s.Format("\tfreq_to_timerval(%i),\r\n\t{\r\n",freq);
		file.Write(s,s.GetLength());
		int t1,t2,frac=0;
		for(int i=0;i<count;) {
			t1 = abs(times[i++])+frac;
			frac = t1%10;
			t1 /= 10;
			if(i<count) {
				t2 = abs(times[i++])+frac;
				frac = t2%10;
				t2 /= 10;
			} else
				t2=0;
			if(i<count)
				s.Format("\t\t{ %i,\t%i },\r\n",t1,t2);
			else
				s.Format("\t\t{ %i,\t0 }\r\n",t1);
			file.Write(s,s.GetLength());
		}
		s.Format("\t}\r\n};\r\n",freq);
		file.Write(s,s.GetLength());
	} else
		return GetLastError();

	return 0;
}

int CIRScopeDlg::WriteWavFile(LPCSTR filename, int count, int *times, UINT freq)
{
	if(!count) return 0;

	CAudio a;

	int err;
	err=a.SetCarrierFrequency(freq);
	if(err) {
		AfxMessageBox("Can not create audio file\nCarrier frequency out of range");
		return err;
	}

	err=a.Open(filename);
	if(err) return err;

	for(int i=0;i<count;) {
		a.Pulse(times[i++]);
	}
	
	a.Close();

	return 0;
}


void CIRScopeDlg::Decode(int count, int *times, short int *counts, 
		UINT freq, int signalIndex, BOOL capture)
{
	const int context_length_out = 9;
	int context_length_in = 2;
	UINT freq_out = freq;
	int last_freq = -1;
	BOOL freq_varies = false;
	int itemsAdded = 0;

	int item=m_Decode.GetItemCount()-1;

	if(0) { //m_GetDecodeVersion) {
		char version[64];
		m_GetDecodeVersion(version);
		TRACE1("Version: %s\n",version);
	}

	if(m_DecodeIR) {
		int* decode_times=new int[count];
		if(!decode_times) return;
		UINT u=0;
		while(u<static_cast<UINT>(count)) {
			decode_times[u]=abs(times[u]);
			++u;
		}

		char protocol[64];
		int context[context_length_out];
		int	device,subdevice,obc,hex[4];
		char misc[64];
		char error[64];

		int start_burst = 0;
		int end_burst = 0; // actually points to burst after last one in frame
		int max_end_burst = 0;
		enum {MinFrame = 2}; // GD: Must agree with definition in DecodeIR

		memset(protocol,0,sizeof(protocol));
		memset(misc,0,sizeof(misc));
		memset(error,0,sizeof(error));
		for (int i=0; i < context_length_out; i++) context[i] = 0;

		GetSelectedDecodes();

		do {
			start_burst = context[0] & 0xFFFFF; // in case nothing better later
			device=obc=-1;
			subdevice = -context_length_out;
			hex[0]=hex[1]=hex[2]=hex[3]=-1;
			*protocol = *misc = *error = 0;

			if (!m_DisableDecoder) {
				// Pass counts as possible 64-bit pointer
				*(__int64 *)(&context[6]) = reinterpret_cast<__int64>(counts);
				m_DecodeIR(context,decode_times,freq,count/2,0,protocol,&device,&subdevice,&obc,hex,misc,error);
			
				if ( (context[0] & 0xFF000000) == 0xFF000000 )
				{
					context_length_in = min(18-(context[0]>>20&0xF), context_length_out);
					start_burst = context[2] & 0xFFFFF;
					// reset context[0] for next call
					context[0] &= 0xFFFFF;
					context[0] |= context[2] & 0xFFF00000;
				}

				if ( context_length_in > 5 )
					end_burst = (context[4] & 0xFFFFF) + (context[5]>>16) + MinFrame;
				else
					end_burst = (context[0] & 0xFFFFF) + (context[1]>>16) + MinFrame;
			}

			CString s;
#if 0
			s.Format("Context: %i %i",context[0],context[1]);
			AfxMessageBox(s);
#endif
#if 0
			TRACE2("Context: %i %i\n",context[0],context[1]);
			TRACE2("Context: %08X %08X\n",context[0],context[1]);
			TRACE("Protocol: %s Device: %i SubDevice:%i OBC: %i\n",protocol,device,subdevice,obc);
			TRACE("Hex: %02X %02X %02X %02X\n",hex[0],hex[1],hex[2],hex[3]);
			TRACE2("Error: %s Misc: %s\n",error,misc);
			TRACE0("\n");
#endif
			// Set to create an <unknown> entry if nothing returned,
			// identified by context values still at zero
			if (	!*protocol 
				&&	context[0] == 0
				&&	context[1] == 0 )
			{
				start_burst = count/2;
			}

			if (start_burst > max_end_burst) {
				// create null entries for skipped bursts, separating entries
				// when there is a gap of more than 200ms
				int split_start = max_end_burst, split_end = 0;
				for (int i=max_end_burst; i<start_burst; i++) {
					if (decode_times[2*i+1] > 200000 || i == start_burst-1) {
						split_end = i+1;
						if ( context_length_in > 7 ) {
							freq_out = GetFreq(split_start, split_end);
							if (last_freq >= 0 && last_freq != freq_out) {
								freq_varies = true;
							}
							last_freq = freq_out;
						}
						s.Format("%i",++item+1);
						m_Decode.InsertItem(item,s);
						m_Decodes.resize(item+1);
						if (context_length_in > 8) {
							m_Decodes[item] = DecodeItem(signalIndex, split_start, split_end-1,
								13, 0, 0);
						}
						else {
							m_Decodes[item] = DecodeItem(signalIndex, split_start, split_end-1);
						}
						s.Format("%i",freq_out);
						m_Decode.SetItemText(item,1,s);
						m_Decodes[item].freq = freq_out;
						s="<unknown>";
						m_Decode.SetItemText(item,2,s);
						m_Decodes[item].protocol = s;
						s=GetNote(split_start, split_end-1);
						m_Decode.SetItemText(item,6,s);
						m_Decodes[item].usernote=s;
						s.Format("%i",split_start);
						m_Decode.SetItemText(item,8,s);
						s.Format("%i",split_end-1);
						m_Decode.SetItemText(item,9,s);
						itemsAdded++;
						split_start = split_end;
					}
				}
			}

			if (end_burst > max_end_burst) {
				max_end_burst = end_burst;
			}

			if(!*protocol) break;

			if ( context_length_in > 7 ) {
				freq_out = context[6];
				if (last_freq >= 0 && last_freq != freq_out) {
					freq_varies = true;
				}
				last_freq = freq_out;
			}
			s.Format("%i",++item+1);
			m_Decode.InsertItem(item,s);
			m_Decodes.resize(item+1);
			++itemsAdded;

			if (context_length_in > 8) {
				m_Decodes[item] = DecodeItem(signalIndex, start_burst, end_burst-1,
					context[8]>>16&0xFF, context[8]&0xFFFF, context[8]>>24);
			}
			else {
				m_Decodes[item] = DecodeItem(signalIndex, start_burst, end_burst-1);
			}

			s.Format("%i",freq_out);
			m_Decodes[item].freq = freq_out;
			m_Decode.SetItemText(item,1,s);
			m_Decode.SetItemText(item,2,protocol);
			m_Decodes[item].protocol = protocol;

			if(device!=-1) {
				if(subdevice==-1)
					s.Format("%i",device);
				else
					s.Format("%i.%i",device,subdevice);
				m_Decode.SetItemText(item,3,s);
				m_Decodes[item].device = s;
			}
			if(obc!=-1) {
				s.Format("%i",obc);
				m_Decode.SetItemText(item,4,s);
				m_Decodes[item].key = s;
			}
			if(hex[0]!=-1) {
				s.Empty();
				CString ss;
				int x=0;
				while(x<4 && hex[x]!=-1) {
					ss.Format(x?" %02X":"%02X",hex[x]);
					s+=ss;
					++x;
				}
				m_Decode.SetItemText(item,5,s);
				m_Decodes[item].hex = s;
			}
			s=GetNote(start_burst, end_burst-1);
			m_Decode.SetItemText(item,6,s);
			m_Decodes[item].usernote=s;
			if(*misc) 
			{
				m_Decode.SetItemText(item,7,misc);
				m_Decodes[item].misc = misc;
			}
			if(*error) 
			{
				m_Decode.SetItemText(item,10,error);
				m_Decodes[item].error = error;
			}
			s.Format("%i",start_burst);
			m_Decode.SetItemText(item,8,s);
			s.Format("%i",end_burst-1);
			m_Decode.SetItemText(item,9,s);
		} while(TRUE);

		if (capture && m_SelectOnCapture) { 	
			// Select added decodes						
			m_viSelected.clear();
			m_viSelected.resize(itemsAdded);
			for (int i=0; i<itemsAdded; i++) {
				m_viSelected[i] = item-itemsAdded+i+1;
			}
		}
		SetSelectedDecodes(0);
		SelectDecode(false);

		m_Signals[signalIndex].freq = freq_varies ? -1 : freq_out;

		delete[] decode_times;
	}

	m_Decode.EnsureVisible(item,FALSE);
}


void CIRScopeDlg::OnView() 
{
	CFileDialog dlg(TRUE,"*.ict",NULL,OFN_HIDEREADONLY|OFN_PATHMUSTEXIST,
		"IR Capture Text Files (*.ict)|*.ict||");

	dlg.m_ofn.lpstrInitialDir = strViewPath;

	if(dlg.DoModal()==IDOK) {
		CString file_name=dlg.GetPathName();
		strViewPath = file_name.Left(dlg.m_ofn.nFileOffset-1);
		OpenICTFile(file_name);
	}
}

void CIRScopeDlg::OnClearDecode() 
{
	m_Decode.DeleteAllItems();
	m_Decode.UpdateWindow();
	m_Decodes.clear();
	GetDlgItem(IDC_DECODE_STRUCT)->SetWindowText("");
	if (m_hExchangeLib) GetDlgItem(IDC_DECODE_IRP)->SetWindowText("");
	CIrScopeWnd::CloseAll();
}

void CIRScopeDlg::OnSize(UINT nType, int cx, int cy) 
{
	CDialog::OnSize(nType, cx, cy);
	UPDATE_EASYSIZE;
}

void CIRScopeDlg::OnSizing(UINT fwSide, LPRECT pRect) 
{
	if (m_hDecodeLib)
	{
		CDialog::OnSizing(fwSide, pRect);
	    EASYSIZE_MINSIZE(620,400,fwSide,pRect);
	}
}
// (520 is the minimum width and 400 the 
// minimum height we want our dialog to have)


BOOL CIRScopeDlg::PreTranslateMessage(MSG* pMsg)
{
	// TODO: Add your specialized code here and/or call the base class
   if (m_hAccel) 
   {
      if (::TranslateAccelerator(m_hWnd, m_hAccel, pMsg)) 
	  {
         return(TRUE);
      }
   }
	return CDialog::PreTranslateMessage(pMsg);
}

int CIRScopeDlg::OnCreate(LPCREATESTRUCT lpCreateStruct)
{
	if (CDialog::OnCreate(lpCreateStruct) == -1)
		return -1;

	// TODO:  Add your specialized creation code here
	HKEY hkey;
	TCHAR sPort[10];
	TCHAR sPath[MAX_PATH];
	DWORD iValue;
	DWORD iSize;
	DWORD dwDisp;
	DWORD iRevision;
	WINDOWPLACEMENT wndpl;
	wndpl.length = sizeof(WINDOWPLACEMENT);	
	TCHAR sValue[MAX_PATH+6];

	m_hDecodeLib=LoadLibrary("DecodeIR.DLL");
	m_hExchangeLib=LoadLibrary("ExchangeIR.dll");
	m_hFTDILib=LoadLibrary("ftd2xx.dll");
	m_hMakeHexLib=LoadLibrary("MakeHex.dll");
	m_bRestore = false;
	AfxInitRichEdit2();

#ifdef FTDITEST
	if (!m_hFTDILib) AfxMessageBox("ftd2xx.dll not found");
#endif

	// get application folder
	if (GetModuleFileName(NULL, sPath, MAX_PATH) > 0)
	{
		m_sAppFolder = CString(sPath);
		m_sAppFolder = m_sAppFolder.Left(m_sAppFolder.ReverseFind((TCHAR)('\\')));
		strSavePath = strViewPath = strExportPath = m_sAppFolder;
	}
	strIRPPath = "";

	// register .ict filetype
 	if (RegCreateKeyEx(HKEY_CLASSES_ROOT, _T(".ict"), 0, NULL,
			REG_OPTION_NON_VOLATILE, KEY_WRITE, NULL, &hkey, 
			&dwDisp) == ERROR_SUCCESS)
	{
		strcpy_s(sValue, sizeof(sValue), "ictfile");
		RegSetValueEx(hkey, _T(""), 0, REG_SZ, (PBYTE)sValue, (DWORD)strlen(sValue)+1);
		RegCloseKey(hkey);
		if (RegCreateKeyEx(HKEY_CLASSES_ROOT, _T("ictfile\\DefaultIcon"), 0, NULL,
			REG_OPTION_NON_VOLATILE, KEY_WRITE, NULL, &hkey, 
			&dwDisp) == ERROR_SUCCESS)
		{
			::GetModuleFileName(AfxGetApp()->m_hInstance, sValue, sizeof(sValue));
			strcat_s(sValue, sizeof(sValue), ",0");
			RegSetValueEx(hkey, _T(""), 0, REG_SZ, (PBYTE)sValue, (DWORD)strlen(sValue)+1);
			RegCloseKey(hkey);
		}
		if (RegCreateKeyEx(HKEY_CLASSES_ROOT, _T("ictfile\\Shell\\Open\\command"), 0, NULL,
			REG_OPTION_NON_VOLATILE, KEY_WRITE, NULL, &hkey, 
			&dwDisp) == ERROR_SUCCESS)
		{
			sValue[0] = '\"';
			::GetModuleFileName(AfxGetApp()->m_hInstance, sValue+1, sizeof(sValue)-1);
			strcat_s(sValue, sizeof(sValue),"\" \"%1\"");
			RegSetValueEx(hkey, _T(""), 0, REG_SZ, (PBYTE)sValue, (DWORD)strlen(sValue)+1);
			RegCloseKey(hkey);
		}
	}

	// Window position only taken from registry if DecodeIR found
	if (RegOpenKeyEx(HKEY_CURRENT_USER, REG_KEY_SET,
			0, KEY_QUERY_VALUE, &hkey) == ERROR_SUCCESS)
	{
		iSize = wndpl.length;
		if (  (m_hDecodeLib) &&
			 RegQueryValueEx(hkey, TEXT("WindowPlacement"), NULL, NULL,
				(LPBYTE)&wndpl, &iSize) == ERROR_SUCCESS )
		{
			RECT wndpos = wndpl.rcNormalPosition;
			if (	wndpos.bottom - wndpos.top > 100
				&&	wndpos.right - wndpos.left > 100 ) 
			{
				m_bRestore = true;
			}
		}
		iSize = sizeof(DWORD);
		if ( RegQueryValueEx(hkey, TEXT("Revision"), NULL, NULL,
					(LPBYTE)&iValue, &iSize ) == ERROR_SUCCESS )
			iRevision = iValue;
		else iRevision = 0;

		// Do not restore window and control positions if revision
		// number has changed
		if (iRevision != REVISION) m_bRestore = false;

		if (m_bRestore)
		{
			wndpl.showCmd = SW_SHOWNORMAL;
			SetWindowPlacement(&wndpl);
		}

		iSize = sizeof(sPort);
		if ( RegQueryValueEx(hkey, TEXT("Port"), NULL, NULL,
					(LPBYTE)sPort, &iSize) == ERROR_SUCCESS
				&&	(char*)sPort[iSize-1] == 0 )	// check zero-terminated
			m_Port = _T(sPort);
		iSize = sizeof(DWORD);
		if ( RegQueryValueEx(hkey, TEXT("Duration"), NULL, NULL,
					(LPBYTE)&iValue, &iSize ) == ERROR_SUCCESS )
			m_Duration = iValue;
		iSize = sizeof(DWORD);
		if ( RegQueryValueEx(hkey, TEXT("Hardware"), NULL, NULL,
					(LPBYTE)&iValue, &iSize ) == ERROR_SUCCESS )
			m_Hardware = iValue;
		iSize = sizeof(sPath);
		if ( RegQueryValueEx(hkey, TEXT("SavePath"), NULL, NULL,
					(LPBYTE)sPath, &iSize ) == ERROR_SUCCESS
				&&	(char*)sPath[iSize-1] == 0 )	// check zero-terminated
			strSavePath = _T(sPath);
		else 
			strSavePath = m_sAppFolder;
		iSize = sizeof(sPath);
		if ( RegQueryValueEx(hkey, TEXT("ExportPath"), NULL, NULL,
					(LPBYTE)sPath, &iSize ) == ERROR_SUCCESS
				&&	(char*)sPath[iSize-1] == 0 )	// check zero-terminated
			strExportPath = _T(sPath);
		else
			strExportPath = m_sAppFolder;
		iSize = sizeof(sPath);
		if ( RegQueryValueEx(hkey, TEXT("IRPPath"), NULL, NULL,
					(LPBYTE)sPath, &iSize ) == ERROR_SUCCESS
				&&	(char*)sPath[iSize-1] == 0 )	// check zero-terminated
			strIRPPath = _T(sPath);
		else
			strIRPPath = "";
		RegCloseKey(hkey);
		strViewPath = strSavePath;
	}
	return 0;
}

void CIRScopeDlg::OnFileOpen()
{
	// TODO: Add your command handler code here
	OnView();
}

void CIRScopeDlg::OnFileSave()
{
	// TODO: Add your command handler code here
	OnFileSaveCommon(SAVE_LAST);
}

void CIRScopeDlg::OnInitMenuPopup(CMenu* pPopupMenu, UINT nIndex, BOOL bSysMenu)
{
//	CDialog::OnInitMenuPopup(pPopupMenu, nIndex, bSysMenu);

	// TODO: Add your message handler code here
	UNREFERENCED_PARAMETER(bSysMenu);
	UNREFERENCED_PARAMETER(nIndex);
	UpdateMenu(pPopupMenu);
}

void CIRScopeDlg::UpdateMenu(CMenu* pMenu)
{
	CCmdUI cmdUI;

	if (NULL != pMenu)
	{
		for (UINT n = 0; n < pMenu->GetMenuItemCount(); ++n)
		{
			CMenu* pSubMenu = pMenu->GetSubMenu(n);
			
			if (NULL != pSubMenu)
			{
				UpdateMenu (pSubMenu);		// recursive call
			}
			else
			{
				cmdUI.m_nIndex = n;
				cmdUI.m_nID = pMenu->GetMenuItemID(n);
				cmdUI.m_pMenu = pMenu;
				// Need to set this value otherwise you will assert when
				// you have an empty recent file list.
				cmdUI.m_nIndexMax = pMenu->GetMenuItemCount();
				cmdUI.DoUpdate(this, FALSE);
			}
		}	
	}
}

void CIRScopeDlg::OpenICTFile(LPCTSTR file_name)
{
		CStdioFile file;

		if(file.Open(file_name,CFile::modeRead|CFile::shareExclusive)) {
			CString s;
			if(!file.ReadString(s) || s!="irscope 0") {
				AfxMessageBox("This is not an IR Scope Capture Text file");
				return;
			}
			UINT u=0;

			m_UserNotes.clear();
			m_times = NULL;	
			m_freq = 0;		
			m_counts = NULL;
			m_count = 0;
			while(file.ReadString(s)) {
				if(s[0]=='+' || s[0]=='-' || isdigit(s[0])) {
					if((u<m_count) && m_times && m_counts) {
						const int comma=s.Find(',');
						if(comma==-1) {
							m_times[u]=atoi(s);
							m_counts[u]=0;
						} else {
							m_times[u]=atoi(s.Left(comma));
							m_counts[u]=atoi(s.Mid(comma+1));
						}
						++u;
					}
				} else if(s.Left(17).CompareNoCase("carrier_frequency")==0) {
					m_freq=atoi(s.Mid(18));
				} else if(s.Left(12).CompareNoCase("sample_count")==0) {
					m_count=atoi(s.Mid(13));
					m_times=new int[m_count];
					m_counts=new short int[m_count];
				} else if(s.Left(5).CompareNoCase("note=")==0) {
					m_UserNotes.push_back(UserNote(u/2,s.Mid(5)));
				} else {
					TRACE1("wtf: %s\n",s);
				}
			}
			if(m_count && m_times && m_counts) {
				UpdateData();

				int item = m_Signals.size();
				m_Signals.resize(item+1);
				m_Signals[item] = Signal(NULL, m_times, m_counts, m_count, m_freq);
				Decode(m_count,m_times,m_counts,m_freq,item,false);

				if (m_ShowWaveform) {
					CString cName = CString(file_name);
					cName = cName.Right(cName.GetLength() - cName.ReverseFind((TCHAR)('\\')) - 1);
					m_currentWnd = CreateWaveform(cName, item);
				}

				AfxGetApp()->AddToRecentFileList(file_name);
			} else m_counts = NULL;			
		} else {
			AfxMessageBox(CString("Could not open file ")+ file_name);
		}
}

LRESULT CIRScopeDlg::OnOpenICTFile(WPARAM wParam, LPARAM lParam)
{
//	return LRESULT();
	UNREFERENCED_PARAMETER(wParam);

	LRESULT rslt = 0;

	OpenICTFile((LPCTSTR) lParam);

	return(rslt);

}

void CIRScopeDlg::SetCommandLine ( LPCTSTR aCommand )
{
	char* tokens[3];
	char cmd[256];
	char* filename;
	char* context = NULL;
	strcpy_s(cmd, sizeof(cmd), aCommand);

	if (cmd[0] == 0) return;
	// get the command
	else if (cmd[0] == '\"') tokens[0] = strtok_s(cmd, "\"", &context);	// this will be the command
	else tokens[0] = strtok_s(cmd, " ", &context);
	// get the filename (with leading spaces) if not quoted
	tokens[1] = strtok_s(NULL, "\"", &context);
	// get the filename if quoted, otherwise this will return null pointer
	tokens[2] = strtok_s(NULL, "\"", &context);
	// select the right filename
	if (tokens[2] == NULL) filename = tokens[1];
	else filename = tokens[2];
	// skip any leading spaces
	for(; *filename == ' '; filename++);
	// tell the app to open the file
	 AfxGetApp()->m_pMainWnd->SendMessage(WM_OPEN_DLG_FILE, 0, (LPARAM)filename);
}

void CIRScopeDlg::OnFileClear()
{
	// Parameter = false means that no deletion takes place,
	// it just returns the max size of the MRU
	int size = ((CIRScopeApp*)AfxGetApp())->DeleteMRU(false);
	// Need to delete items one at a time, updating menu in between,
	// otherwise menu display gets corrupted
	for (int i = 0; i < size; i++)
	{
		((CIRScopeApp*)AfxGetApp())->DeleteMRU(true);
		UpdateMenu(GetMenu());
	}
}

void CIRScopeDlg::OnFileCapture()
{
	// TODO: Add your command handler code here
	OnCapture();
}

void CIRScopeDlg::OnFileSetsavefolder()
{
	// TODO: Add your command handler code here
	BROWSEINFO bi = { 0 };
    bi.lpszTitle = _T("Select the folder for saving files \
or type the full folder path in the box below, starting with the \
drive letter.  If you type, then entering a \\ will give a \
drop-down list showing the contents of that path and allowing \
selection of the next element.");
	bi.ulFlags = BIF_USENEWUI|BIF_VALIDATE|BIF_RETURNONLYFSDIRS;
	bi.lpfn = &SaveCallbackProc;
	bi.hwndOwner = m_hWnd;
    LPITEMIDLIST pidl = SHBrowseForFolder ( &bi );
    if ( pidl != 0 )
    {
        // get the name of the folder
        TCHAR path[MAX_PATH];
        if ( SHGetPathFromIDList ( pidl, path ) )
        {
			strSavePath = path;
			strViewPath = strSavePath;
        }

        // free memory used
        IMalloc * imalloc = 0;
        if ( SUCCEEDED( SHGetMalloc ( &imalloc )) )
        {
            imalloc->Free ( pidl );
            imalloc->Release ( );
        }
    }
}

void CIRScopeDlg::OnAppAbout()
{
	// TODO: Add your command handler code here
	CAboutDlg aDlg;

	aDlg.DoModal();
}

void CIRScopeDlg::OnFileSaveAs()
{
	// TODO: Add your command handler code here
	OnFileSaveAsCommon(SAVE_LAST);
}

void CIRScopeDlg::OnNMClickDecode(NMHDR *pNMHDR, LRESULT *pResult)
{
	LPNMITEMACTIVATE pNMItemActivate = reinterpret_cast<LPNMITEMACTIVATE>(pNMHDR);
	// TODO: Add your control notification handler code here
	SelectDecode(false);
	*pResult = 0;
}


void CIRScopeDlg::SaveICTFile(CString file_name, int mode)
{
	DecodeItem dcItem;
	CString cName = CString(file_name);
	cName = cName.Right(cName.GetLength() - cName.ReverseFind((TCHAR)('\\')) - 1);

	if (mode != SAVE_SELECTED && m_counts == NULL) {
		AfxMessageBox("Nothing to save");
		return;
	} else if (mode == SAVE_SELECTED && m_Decode.GetSelectedCount() == 0) {
		AfxMessageBox("Nothing selected");
		return;
	}
	else if (mode == SAVE_LAST) { 
		WriteFile(file_name,m_count,m_times,m_counts,m_freq);
		if (m_currentWnd != NULL) m_currentWnd->SetWindowText(cName);
	}
	else {
		int totCount=0, burstCount=0, iSize=0;
		UINT averageFreq=0;
		double pulseCount=0., timeCount=0.;
		Signal sig;
		if (mode == SAVE_SELECTED) {
			GetSelectedDecodes();
			iSize = m_viSelected.size();
		} else {
			iSize = m_Decode.GetItemCount();
		}
		for (int i=0; i<iSize; i++)
		{
			dcItem = m_Decodes[(mode==SAVE_SELECTED) ? m_viSelected[i] : i];
			burstCount = dcItem.end - dcItem.start + 1;
			sig = m_Signals[dcItem.signalIndex];
			totCount += 2*burstCount;
			for (int j=2*dcItem.start; j<2*(dcItem.end+1); j++)
			{
				// Don't include case sig.counts[j]=1 as this is an artefact of
				// unmodulated signals in Widget Count mode.
				if (sig.counts !=  NULL && sig.times[j]>0 && sig.counts[j]>1)
				{
					pulseCount += sig.counts[j];
					timeCount += sig.times[j];
				}
			}
		}
		if (timeCount>0) averageFreq = UINT(1000000*pulseCount/timeCount + 0.5);
		for (int i=0; i<iSize; i++)
		{
			dcItem = m_Decodes[(mode==SAVE_SELECTED) ? m_viSelected[i] : i];
			burstCount = dcItem.end - dcItem.start + 1;
			sig = m_Signals[dcItem.signalIndex];
			WriteFile(file_name,2*burstCount,sig.times+2*dcItem.start,
				sig.counts+2*dcItem.start,averageFreq,totCount,i>0,&dcItem.usernote);
		}
	}
	AfxGetApp()->AddToRecentFileList(file_name);
}

UINT CIRScopeDlg::GetFreq(int start, int end)
{
	int s=0;
	int t=0;
	UINT freq;

	if (m_counts == NULL) return 0;

	for (int n=2*start; n<2*end; n++)
	{	
		if (m_times[n] > 0)
		{
			s += m_counts[n];
			t += m_times[n];
		}
	}
	freq=((s!=end-start)&&t)?static_cast<UINT>(static_cast<unsigned __int64>(s)*1000000/t):0;
	return freq;
}

void CIRScopeDlg::OnCbnSelchangeHardware()
{
	// TODO: Add your control notification handler code here

	UpdateData();
	SetAvailablePorts(m_Hardware);
}

void CIRScopeDlg::SetAvailablePorts(int hardware)
{
	CComboBox* cboPort = (CComboBox*)GetDlgItem(IDC_PORT);
	CString sPort;
	int i, num;
	int count = 0;
	LONG ComPort;

	for (i = cboPort->GetCount()-1; i >= 0; i--)
	{
		cboPort->DeleteString( i );
	}
	m_bAutolocated = FALSE;

#ifdef FTDITEST
	if (!m_AutoWidget) AfxMessageBox("Autolocate Widget is turned off");
#endif

	if (m_hFTDILib && m_AutoWidget && (hardware == 0 || hardware == 3)
		&& SetFunctions(m_hFTDILib)
		&& (num = GetNumFTDIDevices()) > 0)	// Widget Count & Time
	{
		for (i = 0; i < num; i++)
		{
			if ( TestFTDIDevice(i, &ComPort) > 0
				&&	ComPort > 0 )
			{
				sPort.Format("COM%i", ComPort);
				cboPort->AddString(sPort);
				count++;
			}
		}
		if (count > 0)
		{
			m_bAutolocated = TRUE;
			if ( cboPort->SelectString(-1, m_Port) == CB_ERR )
			{
				cboPort->SetCurSel(0);
			}
		}
	}

	if (count == 0)
	{
		count = 16;
		for (i = 1; i <= 16; i++)
		{
			sPort.Format("COM%i", i);
			cboPort->AddString(sPort);
		}
		if (cboPort->SelectString(-1, m_Port) == CB_ERR)
		{
			count++;
			cboPort->AddString(m_Port);
			cboPort->SelectString(-1, m_Port);
		}
	}
	cboPort->EnableWindow(count > 1);
}

void CIRScopeDlg::ToggleMenuOption(UINT nID, BOOL* statevar, int ndx, int force)
{
	UpdateData();
	CMenu* menu = GetMenu()->GetSubMenu(OPT_MENU);
	if (ndx >= 0) menu = menu->GetSubMenu(ndx);
	UINT state = force < 0 ? menu->GetMenuState(nID, MF_BYCOMMAND) :
		force == 0 ? MF_CHECKED : MF_UNCHECKED;
	ASSERT(state != 0xFFFFFFFF);
	if (state & MF_CHECKED) {
		menu->CheckMenuItem(nID, MF_UNCHECKED | MF_BYCOMMAND);
	}
	else {
		menu->CheckMenuItem(nID, MF_CHECKED | MF_BYCOMMAND);
	}
	if (statevar != NULL) {
		*statevar = !(state & MF_CHECKED);
	}
	UpdateData(false);
	if ( m_HideSavepanel && (m_SaveICT || m_SaveTVBG || m_SaveWav) ) {
		m_AutoSaveNote.ShowWindow(SW_SHOW);
	}
	else {
		m_AutoSaveNote.ShowWindow(SW_HIDE);
	}
}

void CIRScopeDlg::OnAutotypesIct()
{
	// TODO: Add your command handler code here
	ToggleMenuOption(ID_AUTOTYPES_ICT, &m_SaveICT, 0);
}

void CIRScopeDlg::OnAutotypesWav()
{
	// TODO: Add your command handler code here
	ToggleMenuOption(ID_AUTOTYPES_WAV, &m_SaveWav, 0);
}

void CIRScopeDlg::OnAutotypesTvbg()
{
	// TODO: Add your command handler code here
	ToggleMenuOption(ID_AUTOTYPES_TVBG, &m_SaveTVBG, 0);
}

void CIRScopeDlg::OnAutomodes(UINT nID)
{
	// TODO: Add your command handler code here
	UpdateData();
	CMenu* menu = GetMenu()->GetSubMenu(OPT_MENU)->GetSubMenu(1);
	menu->CheckMenuRadioItem(ID_AUTOMODES_AUTO, ID_AUTOMODES_PROMPT, 
      nID, MF_BYCOMMAND);
	m_AskName = (nID == ID_AUTOMODES_PROMPT);
	UpdateData(false);
	if ( m_HideSavepanel && (m_SaveICT || m_SaveTVBG || m_SaveWav) ) {
		m_AutoSaveNote.ShowWindow(SW_SHOW);
	}
	else {
		m_AutoSaveNote.ShowWindow(SW_HIDE);
	}
}

void CIRScopeDlg::CopyBtnToMenu(UINT btnID, UINT menuID)
{
	CButton* pBtn = (CButton*) GetDlgItem(btnID);
	CMenu* menu = GetMenu()->GetSubMenu(OPT_MENU)->GetSubMenu(0);
	menu->CheckMenuItem(menuID, (pBtn->GetCheck()==BST_UNCHECKED ?
		MF_UNCHECKED : MF_CHECKED)| MF_BYCOMMAND);
	UpdateData();
	if ( m_HideSavepanel && (m_SaveICT || m_SaveTVBG || m_SaveWav) ) {
		m_AutoSaveNote.ShowWindow(SW_SHOW);
	}
	else {
		m_AutoSaveNote.ShowWindow(SW_HIDE);
	}
}

void CIRScopeDlg::OnBnClickedSaveIct()
{
	// TODO: Add your control notification handler code here
	CopyBtnToMenu(IDC_SAVE_ICT, ID_AUTOTYPES_ICT);
}

void CIRScopeDlg::OnBnClickedSaveWav()
{
	// TODO: Add your control notification handler code here
	CopyBtnToMenu(IDC_SAVE_WAV, ID_AUTOTYPES_WAV);
}

void CIRScopeDlg::OnBnClickedSaveTvbg()
{
	// TODO: Add your control notification handler code here
	CopyBtnToMenu(IDC_SAVE_TVBG, ID_AUTOTYPES_TVBG);
}

void CIRScopeDlg::OnBnClickedAskName()
{
	// TODO: Add your control notification handler code here
	CButton* pBtn = (CButton*) GetDlgItem(IDC_ASK_NAME);
	OnAutomodes(pBtn->GetCheck()==BST_UNCHECKED ? ID_AUTOMODES_AUTO : ID_AUTOMODES_PROMPT);
}

void CIRScopeDlg::OnAutowidget()
{
	// TODO: Add your command handler code here
	ToggleMenuOption(ID_AUTOWIDGET, &m_AutoWidget, -1);
	SetAvailablePorts(m_Hardware);
}

void CIRScopeDlg::OnOptionsNobeeps()
{
	// TODO: Add your command handler code here
	ToggleMenuOption(ID_OPTIONS_NOBEEPS, &m_NoBeeps, -1);
}

void CIRScopeDlg::OnOptionsHidesavepanel()
{
	// TODO: Add your command handler code here
	ToggleMenuOption(ID_OPTIONS_HIDESAVEPANEL, &m_HideSavepanel, -1);
	int swState = m_HideSavepanel ? SW_HIDE : SW_SHOW;
	GetDlgItem(IDC_STATIC_SAVEBOX)->ShowWindow(swState);
	GetDlgItem(IDC_SAVE_ICT)->ShowWindow(swState);
	GetDlgItem(IDC_SAVE_WAV)->ShowWindow(swState);
	GetDlgItem(IDC_SAVE_TVBG)->ShowWindow(swState);
	GetDlgItem(IDC_ASK_NAME)->ShowWindow(swState);
}

void CIRScopeDlg::OnNMRClickDecode(NMHDR *pNMHDR, LRESULT *pResult)
{
	LPNMITEMACTIVATE pNMItemActivate = reinterpret_cast<LPNMITEMACTIVATE>(pNMHDR);
	// TODO: Add your control notification handler code here

	CMenu menuSelect;
	CMenu* menuSub;
	CPoint point;
	int res;
	GetCursorPos(&point);
	menuSelect.LoadMenu(IDR_MENU_SELECT);
	menuSub = menuSelect.GetSubMenu(0);

	if (m_Decode.GetSelectedCount() == 0) {
		menuSelect.EnableMenuItem(ID_SELECT_ADDNOTE, MF_GRAYED|MF_BYCOMMAND);
		menuSelect.EnableMenuItem(ID_SELECT_EXPORT, MF_GRAYED|MF_BYCOMMAND);
		menuSelect.EnableMenuItem(ID_SELECT_DELETE, MF_GRAYED|MF_BYCOMMAND);
	}

	res = menuSub->TrackPopupMenu(TPM_LEFTALIGN | TPM_LEFTBUTTON | TPM_RETURNCMD,
        point.x, point.y, this);
	switch (res) {
	case ID_SELECT_SELECTALL:
		SelectAll();
		break;
	case ID_SELECT_DESELECTALL:
		DeselectAll();
		break;
	case ID_SELECT_ADDNOTE:
		OnBnClickedBnNote();
		break;
	case ID_SELECT_EXPORT:
		OnBnClickedBnExport();
		break;
	case ID_SELECT_DELETE:
		OnBnClickedBnDelete();
		break;
	case ID_SELECT_CLEARALL:
		OnClearDecode();
		break;
	}

	menuSelect.DestroyMenu();
	*pResult = 0;
}

/*
void CIRScopeDlg::OnFileSaveSelected()
{
	// TODO: Add your command handler code here
	OnFileSaveCommon(SAVE_SELECTED);
}
*/

void CIRScopeDlg::OnFileSaveSelectedAs()
{
	// TODO: Add your command handler code here
	OnFileSaveAsCommon(SAVE_SELECTED);
}

void CIRScopeDlg::OnFileSaveCommon(int mode)
{
	CString file_name = "";
	if (strSavePath != _T("")) file_name = strSavePath+"\\";
	file_name += CTime::GetCurrentTime().Format("IR%Y%m%d%H%M%S")+".ict";
	SaveICTFile(file_name, mode);
}

void CIRScopeDlg::OnFileSaveAsCommon(int mode)
{
	// TODO: Add your command handler code here

	if (mode != SAVE_SELECTED && m_counts == NULL) {
		AfxMessageBox("Nothing to save");
		return;
	} else if (mode == SAVE_SELECTED && m_Decode.GetSelectedCount() == 0) {
		AfxMessageBox("Nothing selected");
		return;
	}

	CString file_name(CTime::GetCurrentTime().Format("IR%Y%m%d%H%M%S")+".ict");
	if (m_counts == NULL) MessageBox("Nothing to save");
	else {
		CFileDialog dlg(FALSE,"*.ict",file_name,OFN_HIDEREADONLY|OFN_PATHMUSTEXIST,
			"IR Capture Text Files (*.ict)|*.ict||");

		dlg.m_ofn.lpstrInitialDir = strViewPath;
		dlg.m_ofn.lpstrTitle = (mode == SAVE_LAST) ? "Save Last Signal As" :
			(mode == SAVE_SELECTED) ? "Save Selected As" :
			(mode == SAVE_ALL) ? "Save All As" : "Save As";

		if(dlg.DoModal()==IDOK) {
			file_name=dlg.GetPathName();
			strViewPath = file_name.Left(dlg.m_ofn.nFileOffset-1);
			SaveICTFile(file_name, mode);
		}
	}
}

void CIRScopeDlg::DeselectAll()
{
	LVITEM lvi;
	lvi.mask = LVIF_STATE; 
	lvi.state = 0; 
	lvi.stateMask = LVIS_SELECTED|LVIS_FOCUSED;

	for (unsigned int i=0; i<m_Decodes.size(); i++)
		m_Decode.SetItemState(i, &lvi);
	GetDlgItem(IDC_DECODE_STRUCT)->SetWindowText("");
	if (m_hExchangeLib) GetDlgItem(IDC_DECODE_IRP)->SetWindowText("");
	ClearRedline();
}

void CIRScopeDlg::OnHelpDecodeirHtml()
{
	// TODO: Add your command handler code here
	CString exename = GetExeByExtension(".html");
	CString htmlname = m_sAppFolder + "\\DecodeIR.html";
	DWORD err;
	if (exename == _T("")) {
		AfxMessageBox("Unable to locate HTML reader");
		return;
	}
	err = GetFileAttributes(htmlname);
	if (err == INVALID_FILE_ATTRIBUTES || (FILE_ATTRIBUTE_DIRECTORY & err) != 0) {
		AfxMessageBox("DecodeIR.html missing");
		return;
	}
	if ((__int64)ShellExecute(NULL, NULL, exename, htmlname, NULL, SW_SHOWNORMAL) <= 32)
	{
		AfxMessageBox("Unidentified error in opening DecodeIR.html");
	}
}

void CIRScopeDlg::SelectDecode(BOOL bRedLine)
{
	int row;
	CIrScopeWnd *wave_wnd = NULL;
	DecodeItem dcItem;
	int start_time = 0;

	ClearRedline();

	if (m_Decode.GetSelectedCount() == 1)
	{
		CString note;
		GetSelectedDecodes();
		row = m_viSelected[0];
		dcItem = m_Decodes[row];

		if (dcItem.note > 0 && !m_DisableDecoder)
		{
			if (dcItem.note == 2 && dcItem.param == 1) dcItem.note = 1;
			else if (dcItem.note == 12 && dcItem.param == 1) dcItem.note = 1;
			else if (dcItem.note == 3 && dcItem.param == 1) dcItem.note = 10;
			else if (dcItem.note == 4 && dcItem.param == 1) dcItem.note = 11;
			else if (dcItem.note == 5 && dcItem.param == 1)
			{	
				dcItem.note = 1;
				dcItem.auxNote = 6;
			}
			else if (dcItem.note == 6 && dcItem.param == 1)
			{	
				dcItem.note = 1;
				dcItem.auxNote = 7;
			}

			note.Format(m_Notes[dcItem.note-1], dcItem.param);
			if (dcItem.auxNote > 0)
				note += m_AuxNotes[dcItem.auxNote-1];
			GetDlgItem(IDC_DECODE_STRUCT)->SetWindowText(note);
		} else {
			GetDlgItem(IDC_DECODE_STRUCT)->SetWindowText("");
		}
		
		if (m_hExchangeLib) {
			CString sIRP = MakeIRP(dcItem);
			if ( sIRP.IsEmpty() ) sIRP = CString("Undetermined");
			GetDlgItem(IDC_DECODE_IRP)->SetWindowText(sIRP);
		}

		UpdateData();

		if (bRedLine)
		{
			int index = dcItem.signalIndex;
			wave_wnd = (CIrScopeWnd*)m_Signals[index].wnd;
			if (wave_wnd == NULL) wave_wnd = CreateWaveform("IR Scope", index);
			wave_wnd->SetHighlightRange(2*dcItem.start, 2*dcItem.end+1);
			wave_wnd->Invalidate();
			wave_wnd->SetFocus();
		}
	}
	else
	{
		GetDlgItem(IDC_DECODE_STRUCT)->SetWindowText("");
		if (m_hExchangeLib) GetDlgItem(IDC_DECODE_IRP)->SetWindowText("");
		if (m_Decode.GetSelectedCount() == 0) DefocusAll(); 
	}
}

void CIRScopeDlg::OnThumbScale(UINT nID)
{
	// TODO: Add your command handler code here

	CMenu* menu = GetMenu()->GetSubMenu(OPT_MENU)->GetSubMenu(7);
	menu->CheckMenuRadioItem(ID_THUMB_FIXEDRES, ID_THUMB_ENTIREWAVE, 
      nID, MF_BYCOMMAND);

	switch (nID)
	{
	case ID_THUMB_FIXEDRES:
		m_iThumbScale = 0;
		break;
	case ID_THUMB_LIMITEDDUR:
		m_iThumbScale = 1;
		break;
	case ID_THUMB_ENTIREWAVE:
		// run through to default
	default:
		m_iThumbScale = 2;
	}

	DecodeItem dcItem;
	CIrScopeWnd *wave_wnd = NULL;
	for (unsigned int i=0; i<m_Signals.size(); i++)
	{
		wave_wnd = (CIrScopeWnd*)m_Signals[i].wnd;
		if (wave_wnd != NULL) wave_wnd->Invalidate();
	}
}

void CIRScopeDlg::OnFileSetexportfolder()
{
	// TODO: Add your command handler code here
	BROWSEINFO bi = { 0 };
    bi.lpszTitle = _T("Select the folder for saving Export output \
or type the full folder path in the box below, starting with the \
drive letter.  If you type, then entering a \\ will give a \
drop-down list showing the contents of that path and allowing \
selection of the next element.");
	bi.ulFlags = BIF_USENEWUI|BIF_VALIDATE|BIF_RETURNONLYFSDIRS;
	bi.lpfn = &ExportCallbackProc;
	bi.hwndOwner = m_hWnd;
    LPITEMIDLIST pidl = SHBrowseForFolder ( &bi );
    if ( pidl != 0 )
    {
        // get the name of the folder
        TCHAR path[MAX_PATH];
        if ( SHGetPathFromIDList ( pidl, path ) )
        {
			strExportPath = path;
        }

        // free memory used
        IMalloc * imalloc = 0;
        if ( SUCCEEDED( SHGetMalloc ( &imalloc )) )
        {
            imalloc->Free ( pidl );
            imalloc->Release ( );
        }
    }
}

void CIRScopeDlg::OnBnClickedBnExport()
{
	OnExportCommon(true);
	m_Decode.SetFocus(); 
}

void CIRScopeDlg::OnExportCommon(BOOL bSelected)
{
	// TODO: Add your control notification handler code here
	Signal sig;
	int row;

	if (! bSelected)
	{
		if (m_counts == NULL) 
		{	
			AfxMessageBox("Nothing to export");
			return;
		}
		else 
		{
			// row=INT_MAX signifies the most recent signal
			ExportDecode(INT_MAX);
		}
	}
	else if (m_Decode.GetSelectedCount() == 0)
	{
		AfxMessageBox("Nothing selected");
		return;
	}
	else for (row = 0; row < m_Decodes.size(); row++)
	{
		if ( m_Decode.GetItemState(row, LVIS_SELECTED) )
		{
			ExportDecode(row);
		}
	}
	CString s = "Selected signals exported as ";
	switch (m_iExportMode)
	{
	case 0:
		s += "UEI Learned.";
		break;
	case 1:
		s += "Pronto from signal.";
		break;
	case 2:
		s += "Pronto from decode.";
		break;
	case 3:
		s += "Lintronic.";
		break;
	default:
		s += "unknown format.";
	}
	s += "\n\nUse Export Mode on the Export menu\nto change the format.";
	AfxMessageBox(s, MB_OK | MB_ICONINFORMATION);

}

void CIRScopeDlg::ExportDecode(size_t row)
{
	CString filename = m_CommonFolder ? strSavePath+"\\" : strExportPath+"\\";
	CString strID = "", IRPname="", FullIRPname="", errStr = "";
	int* signal_out = NULL;
	int bursts[2*16];
	int bursts_size = 0;
	int sngl_count = 0, rpt_count = 0, extra_count = 0, rpts = 0, errCode = 0;
	int freq;
	int* times;
	DecodeItem dcItem;
	if (row != INT_MAX) {
		dcItem = m_Decodes[row];
		Signal sig = m_Signals[dcItem.signalIndex];
		sngl_count = dcItem.end-dcItem.start+1;
		freq = dcItem.freq;
		times = sig.times+2*dcItem.start;
		strID.Format("%s%s%s%s",
			dcItem.usernote != "" ? dcItem.usernote + " : " : "",
			"Protocol="+dcItem.protocol,
			dcItem.device != "" ? " Device="+dcItem.device : "",
			dcItem.key != "" ? " OBC="+dcItem.key : "");
	} else {
		sngl_count = m_count/2;
		freq = m_freq;
		times = m_times;
		strID = "Export of entire signal before decoding";
	}


	// We need to allow timing errors that are due to the way IRScope works, as well as
	// those in the original signal so set errlimit to be 2.5 times carrier cycle time
	// in microseconds
	m_FindRepeat(times, sngl_count, 
		rpt_count, extra_count, rpts, GetErrlimit(freq));	

	delete [] signal_out;
	signal_out = new int[2*(sngl_count+rpt_count+extra_count)];

	switch (m_iExportMode)
	{
	case 0:
		filename += "ExportedUEI.txt";
		strID += " (UEI Learned)";
		bursts_size = m_Analyze(times, sngl_count, rpt_count, 
			extra_count, rpts, signal_out, bursts, 16, GetErrlimit(freq), -1, NULL);
		errCode = m_MakeUEILearned(filename.GetBuffer(0), strID.GetBuffer(0), 
			signal_out, sngl_count, rpt_count, extra_count, bursts, bursts_size, freq, NULL);
		if (errCode <= 0) {
			errStr = (errCode == 0) ? "Unable to open output file." :
				(errCode < 0 && errCode > -4) ? "Export error.  See output file for details." :
				"Unknown error in export process.";;
			AfxMessageBox(errStr);
		}
		break;
	case 1:
		filename += "ExportedPronto.txt";
		strID += " (Pronto from signal)";
		m_Analyze(times, sngl_count, rpt_count, 
			extra_count, rpts, signal_out, NULL, 0, GetErrlimit(freq),-1,NULL);
		errCode = m_MakePronto(filename.GetBuffer(0), strID.GetBuffer(0), 
			signal_out, freq, sngl_count, rpt_count, NULL);
		if (errCode <= 0) {
			errStr = (errCode == 0) ? "Unable to open output file." :
				"Unknown error in export process.";
			AfxMessageBox(errStr);
		}
		break;
	case 2:
		filename += "ExportedPronto.txt";
		if ( strIRPPath == "") {
			AfxMessageBox("No path set for IRP files.  Please use the \
						  File/Set Folders menu item to set this path.");
			break;
		}			  
		IRPname = ProtocolToIRP(dcItem.protocol)+".irp";
		FullIRPname = strIRPPath+"\\"+IRPname;
		strID += " (Pronto from decode)";
		errCode = m_MakeHex(filename.GetBuffer(0), FullIRPname.GetBuffer(0), 
			strID.GetBuffer(0), dcItem.device.GetBuffer(0), dcItem.key.GetBuffer(0));
		if (errCode <= 0) {
			errStr = (errCode == 0) ? "Unable to open output file." :
				(errCode == -1) ? "IRP file "+IRPname+" not found.":
				(errCode == -2) ? "Error reading IRP file "+IRPname+"." :
				"Unknown error in export process.";
			AfxMessageBox(errStr);
		}
		break;
	case 3:
		filename += "ExportedLintronic.txt";
		strID += " (Lintronic)";
		m_Analyze(times, sngl_count, rpt_count, 
			extra_count, rpts, signal_out, NULL, 0, GetErrlimit(freq),-1,NULL);
		errCode = m_MakeLintronic(filename.GetBuffer(0), strID.GetBuffer(0), 
			signal_out, sngl_count, rpt_count, extra_count, rpts);
		if (errCode <= 0) {
			errStr = (errCode == 0) ? "Unable to open output file." :
				"Unknown error in export process.";
			AfxMessageBox(errStr);
		}
		break;
	default:
		AfxMessageBox("Unknown export mode");
	}
	delete [] signal_out;
	signal_out = NULL;
	m_Bursts.clear();
}

void CIRScopeDlg::OnExportExport()
{
	// TODO: Add your command handler code here
	if (m_iExportMode == 2) {
		AfxMessageBox("Only selected decodes can be exported as Pronto from decode.");
		return;
	}
	OnExportCommon(false);
}

void CIRScopeDlg::OnExportExportselected()
{
	// TODO: Add your command handler code here
	OnExportCommon(true);
}

void CIRScopeDlg::OnExportMode(UINT nID)
{
	// TODO: Add your command handler code here

	CMenu* menu = GetMenu()->GetSubMenu(EXPORT_MENU)->GetSubMenu(3);
	menu->CheckMenuRadioItem(ID_EXPORTMODE_UEILEARNED, ID_EXPORTMODE_LINTRONIC, 
      nID, MF_BYCOMMAND);

	switch (nID)
	{
	case ID_EXPORTMODE_UEILEARNED:
		m_iExportMode = 0;
		break;
	case ID_EXPORTMODE_PRONTOSIG:
		m_iExportMode = 1;
		break;
	case ID_EXPORTMODE_PRONTODEC:
		m_iExportMode = 2;
		break;
	case ID_EXPORTMODE_LINTRONIC:
		m_iExportMode = 3;
		break;
	default:
		m_iExportMode = 0;
	}
}

void CIRScopeDlg::DeleteDecode()
{
	int row = 0, index = 0;
	ClearRedline();

	if (m_Decode.GetSelectedCount() != 0)
	{
		for (row = m_Decode.GetItemCount()-1; row >= 0; row--)
		{
			if ( m_Decode.GetItemState(row, LVIS_SELECTED) )
			{
				// Close the waveform window and free signal memory if 
				// they are not used by any other DecodeItem
				bool doFreeSignal = true;;
				index = m_Decodes[row].signalIndex;
				for (int i = 0; i < m_Decodes.size(); i++)
				{
					if (i != row && m_Decodes[i].signalIndex == index)
					{	
						doFreeSignal = false; 
						break; 
					}
				}
				if ( doFreeSignal )
				{
					if ( m_Signals[index].wnd != NULL ) 
					{
						m_Signals[index].wnd->DestroyWindow();
					}
					m_Signals[index].done();
				}
				m_Decodes.erase(m_Decodes.begin()+row);
			}
		}
		RefreshDecode();
	}
}

void CIRScopeDlg::OnNMDblclkDecode(NMHDR *pNMHDR, LRESULT *pResult)
{
	LPNMITEMACTIVATE pNMItemActivate = reinterpret_cast<LPNMITEMACTIVATE>(pNMHDR);
	// TODO: Add your control notification handler code here
	SelectDecode(true);
	*pResult = 0;
}

void CIRScopeDlg::OnBnClickedBnDelete()
{
	// TODO: Add your control notification handler code here
	DeleteDecode();
	m_Decode.SetFocus();
}

void CIRScopeDlg::OnBnClickedBnNote()
{
	// TODO: Add your control notification handler code here
	int count = GetSelectedDecodes();
	CIRNoteEdit dlg(this);
	if (count > 0) dlg.DoModal();
	else AfxMessageBox("Nothing selected");
	m_Decode.SetFocus();
}

void CIRScopeDlg::RefreshDecode()
{
	DecodeItem dcItem;
	CString s;
	m_Decode.DeleteAllItems();
	for (int row = 0; row < m_Decodes.size(); row++)
	{
		dcItem = m_Decodes[row];
		s.Format("%i",row+1);
		m_Decode.InsertItem(row, s);
		s.Format("%i", dcItem.freq);
		m_Decode.SetItemText(row,1,s);
		m_Decode.SetItemText(row,2,dcItem.protocol);
		m_Decode.SetItemText(row,3,dcItem.device);
		m_Decode.SetItemText(row,4,dcItem.key);
		m_Decode.SetItemText(row,5,dcItem.hex);
		m_Decode.SetItemText(row,6,dcItem.usernote);
		m_Decode.SetItemText(row,7,dcItem.misc);
		m_Decode.SetItemText(row,10,dcItem.error);
		s.Format("%i",dcItem.start);
		m_Decode.SetItemText(row,8,s);
		s.Format("%i",dcItem.end);
		m_Decode.SetItemText(row,9,s);
	}
}

void CIRScopeDlg::OnBnClickedBnUp()
{
	// TODO: Add your control notification handler code here
	int count = GetSelectedDecodes();
	if (count == 0 || m_viSelected[0] == 0)
	{ 
		m_Decode.SetFocus(); 
		return; 
	}

	int i;
	vector<DecodeItem> sorted(m_Decodes);
	for (i=count-1; i>=0; i--) sorted.erase(sorted.begin()+m_viSelected[i]);
	for (i=0; i<count; i++) sorted.insert(sorted.begin()+m_viSelected[i]-1,
		m_Decodes[m_viSelected[i]]);
	m_Decodes.swap(sorted);
	RefreshDecode();
	SetSelectedDecodes(-1);
}

int CIRScopeDlg::GetSelectedDecodes()
{
	int i = 0;
	int count = m_Decode.GetSelectedCount();
	m_viSelected.clear();
	m_viSelected.resize(count);
	for (int row = 0; row < m_Decode.GetItemCount(); row++)
		if ( m_Decode.GetItemState(row, LVIS_SELECTED) )
			m_viSelected[i++] = row;
	return count;
}

void CIRScopeDlg::OnBnClickedBnDown()
{
	// TODO: Add your control notification handler code here
	int count = GetSelectedDecodes();
	if (count == 0 || m_viSelected[count-1] == m_Decode.GetItemCount()-1) 
	{ 
		m_Decode.SetFocus(); 
		return;
	}

	int i;
	vector<DecodeItem> sorted(m_Decodes);
	for (i=count-1; i>=0; i--) sorted.erase(sorted.begin()+m_viSelected[i]);
	for (i=0; i<count; i++) sorted.insert(sorted.begin()+m_viSelected[i]+1,
		m_Decodes[m_viSelected[i]]);
	m_Decodes.swap(sorted);
	RefreshDecode();
	SetSelectedDecodes(1);
}

CString CIRScopeDlg::GetNote(int start, int end)
{
	CString s = "";
	int iBurst;
	for (int i = 0; i < m_UserNotes.size(); i++)
	{
		iBurst = m_UserNotes[i].burst;
		if ( iBurst >= start && iBurst <= end )
		{
			s = m_UserNotes[i].note;
			break;
		}
	}
	return s;
}

void CIRScopeDlg::OnBnClickedBnSave()
{
	// TODO: Add your control notification handler code here
	OnFileSaveCommon(SAVE_SELECTED);
	m_Decode.SetFocus(); 
}

void CIRScopeDlg::OnBnClickedBnSaveas()
{
	// TODO: Add your control notification handler code here
	OnFileSaveAsCommon(SAVE_SELECTED);
	m_Decode.SetFocus(); 
}

void CIRScopeDlg::SelectAll(void)
{
	LVITEM lvi;
	lvi.mask = LVIF_STATE; 
	lvi.state = LVIS_SELECTED; 
	lvi.stateMask = LVIS_SELECTED;
	ClearRedline();
	GetDlgItem(IDC_DECODE_STRUCT)->SetWindowText(""); // Clear structure window
	if (m_hExchangeLib) GetDlgItem(IDC_DECODE_IRP)->SetWindowText("");
	for (int i=0; i<m_Decode.GetItemCount(); i++) m_Decode.SetItemState(i, &lvi);
	if (m_Decode.GetSelectedCount() == 1) SelectDecode(false);
}

void CIRScopeDlg::SetSelectedDecodes(int offset)
{
	LVITEM lvi;
	int count = m_viSelected.size();
	lvi.mask = LVIF_STATE; 
	lvi.state = LVIS_SELECTED; 
	lvi.stateMask = LVIS_SELECTED;
	DeselectAll();
	for (int i=0; i<count; i++) m_Decode.SetItemState(m_viSelected[i]+offset, &lvi);
	m_Decode.SetFocus();
}

void CIRScopeDlg::DefocusAll()
{
	LVITEM lvi;
	lvi.mask = LVIF_STATE; 
	lvi.state = 0; 
	lvi.stateMask = LVIS_FOCUSED;
	for (unsigned int i=0; i<m_Decode.GetItemCount(); i++)
		m_Decode.SetItemState(i, &lvi);
}

void CIRScopeDlg::OnFileSetirpfolder()
{
	// TODO: Add your command handler code here
		// TODO: Add your command handler code here
	BROWSEINFO bi = { 0 };
    bi.lpszTitle = _T("Select the folder that contains IRP files \
or type the full folder path in the box below, starting with the \
drive letter.  If you type, then entering a \\ will give a \
drop-down list showing the contents of that path and allowing \
selection of the next element.");
	bi.ulFlags = BIF_USENEWUI|BIF_VALIDATE|BIF_RETURNONLYFSDIRS;
	bi.lpfn = &IRPCallbackProc;
	bi.hwndOwner = m_hWnd;
    LPITEMIDLIST pidl = SHBrowseForFolder ( &bi );
    if ( pidl != 0 )
    {
        // get the name of the folder
        TCHAR path[MAX_PATH];
        if ( SHGetPathFromIDList ( pidl, path ) )
        {
			strIRPPath = path;
        }

        // free memory used
        IMalloc * imalloc = 0;
        if ( SUCCEEDED( SHGetMalloc ( &imalloc )) )
        {
            imalloc->Free ( pidl );
            imalloc->Release ( );
        }
    }
}

CString CIRScopeDlg::ProtocolToIRP(CString protocol)
{
	int n = protocol.Find("{");
	CString str = protocol;
	if (n > 0) str = protocol.Left(n);
	str.Replace(".", NULL);
	str.Replace(" ", "_");
	if (str.Left(3) == "X10") str = "x10ir";
	return str;
}

void CIRScopeDlg::ClearRedline(void)
{
	DecodeItem dcItem;
	CIrScopeWnd *wave_wnd = NULL;

	for (unsigned int i=0; i<m_Signals.size(); i++)
	{
		wave_wnd = (CIrScopeWnd*)m_Signals[i].wnd;
		if (wave_wnd != NULL)
		{
			wave_wnd->SetHighlightRange(0, 0);
			wave_wnd->Invalidate();
		}
	}
}

void CIRScopeDlg::OnOptionsSelectOnCapture()
{
	// TODO: Add your command handler code here
	ToggleMenuOption(ID_OPTIONS_SELECTONCAPTURE, &m_SelectOnCapture, -1);
}

void CIRScopeDlg::OnFileSummary()
{
	// TODO: Add your command handler code here
	CIRSummary dlg(this);
	dlg.DoModal();
}


void CIRScopeDlg::OnBnClickedBnImport()
{
	// TODO: Add your control notification handler code here

	int item;
	CIRImport dlg(this);
	if (dlg.DoModal() == IDOK) {
		item = m_Signals.size();
		m_Signals.resize(item+1);
		m_Signals[item] = Signal(NULL, m_times, m_counts, m_count, m_freq);
		Decode(m_count,m_times,m_counts,m_freq,item,true);
		UpdateData(true);
		if (m_ShowWaveform) {
			m_currentWnd = CreateWaveform("IR Scope", item);
		}
	}
}


CIrScopeWnd* CIRScopeDlg::CreateWaveform(CString sName, int sigIndex)
{
	CIrScopeWnd* wnd =new CIrScopeWnd;
	wnd->Create(::GetDesktopWindow(), sName);
	m_Signals[sigIndex].wnd = wnd;
	wnd->SetData(sigIndex, this);
	int freq = m_Signals[sigIndex].freq;
	if( freq > 0) {
		CString fs;
		fs.Format("%i Hz",freq);
		wnd->SetText(fs);
	} else if ( freq < 0) {
		wnd->SetText(CString("<freq varies>"));
	}
	wnd->UpdateWindow();
	return wnd;
}

void CIRScopeDlg::OnOptionsCommonfolder()
{
	// TODO: Add your command handler code here
	ToggleMenuOption(ID_OPTIONS_COMMONFOLDER, &m_CommonFolder, -1);
	CMenu* menu = GetMenu()->GetSubMenu(0)->GetSubMenu(SETFOLDER_SUBMENU);
	menu->EnableMenuItem(ID_FILE_SETEXPORTFOLDER, 
		(m_CommonFolder ? MF_GRAYED : MF_ENABLED) | MF_BYCOMMAND) ;
}

int CIRScopeDlg::GetErrlimit(int freq)
{
	// IRScope cannot directly handle zero frequency, so it will only come as import.
	// Protocol pid-002A has a separation of 50us between distinct SPACE durations, so
	// the error figure is here taken as half that.
	return (freq > 0) ? int(ceil(2500000./freq)) : 25;
}


void CIRScopeDlg::OnOptionsDisabledecoder()
{
	// TODO: Add your command handler code here
	ToggleMenuOption(ID_OPTIONS_DISABLEDECODER, &m_DisableDecoder, -1);
}

void CIRScopeDlg::OnIrpdataformat(UINT nID)
{
	// TODO: Add your command handler code here

	CMenu* menu = GetMenu()->GetSubMenu(OPT_MENU)->GetSubMenu(10);
	menu->CheckMenuRadioItem(ID_IRPDATAFORMAT_BINARY, ID_IRPDATAFORMAT_HEXS, 
      nID, MF_BYCOMMAND);

	switch (nID)
	{
	case ID_IRPDATAFORMAT_BINARY:
		m_iIRPformat = 0;
		break;
	case ID_IRPDATAFORMAT_QUATERNARY:
		m_iIRPformat = 1;
		break;
	case ID_IRPDATAFORMAT_HEXX:
		m_iIRPformat = 2;
		break;
	case ID_IRPDATAFORMAT_HEXS:
		m_iIRPformat = 3;
		break;
	default:
		m_iIRPformat = 3;
	}
}

CString CIRScopeDlg::MakeIRP(DecodeItem dcItem)
{	
	char irp[500];	 
	Signal sig = m_Signals[dcItem.signalIndex];
	int sngl_count = dcItem.end-dcItem.start+1;
	int rpt_count = 0, extra_count = 0, rpts = 0;
	m_FindRepeat(sig.times+2*dcItem.start, sngl_count, 
		rpt_count, extra_count, rpts, GetErrlimit(dcItem.freq));
	int* signal_out = new int[2*(sngl_count+rpt_count+extra_count)];
	irp[0] = "bqx$"[m_iIRPformat];
	m_Analyze(sig.times+2*dcItem.start, sngl_count, rpt_count, 
		extra_count, rpts, signal_out, NULL, 0, GetErrlimit(dcItem.freq),
		dcItem.freq, irp);
	delete [] signal_out;
	return CString(irp);
}

void CIRScopeDlg::OnBnClickedSaveallas()
{
	// TODO: Add your control notification handler code here
	OnFileSaveAsCommon(SAVE_ALL);
	m_Decode.SetFocus();
}

void CIRScopeDlg::OnFileSaveallas()
{
	// TODO: Add your command handler code here
	OnFileSaveAsCommon(SAVE_ALL);
}
